<?php

namespace App\Http\Controllers\Api;

use App\Events\NewOrderPlaced;
use App\Http\Controllers\Controller;
use App\Models\Cart;
use App\Models\CartItems;
use App\Models\Ingredient;
use App\Models\MenuItems;
use App\Models\Sale;
use App\Models\SoldItem;
use App\Models\Table;
use App\Models\Waiter;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class QRController extends Controller
{
    public function receiveOrder(Request $request)
    {
        DB::beginTransaction();

        try {
            $data = $request->all();

            // ✅ Validate required fields
            if (!isset($data['tableId']) || !isset($data['cart']) || !isset($data['items'])) {
                return response()->json([
                    'success' => false,
                    'message' => 'Required fields are missing.',
                    'missing_fields' => array_filter([
                        'tableId' => $data['tableId'] ?? null,
                        'cart' => $data['cart'] ?? null,
                        'items' => $data['items'] ?? null,
                    ], fn($val) => is_null($val)),
                ], 422);
            }

            $tableId = $data['tableId'];
            $cartData = $data['cart'];
            $cartItems = $data['items'];

            // ✅ Check for existing submitted cart for the table
            $cart = Cart::where('table_id', $tableId)
                ->where('status', 'submitted')
                ->first();

            if (!$cart) {
                // Create a new cart if none exists
                $cart = Cart::create([
                    'table_id' => $tableId,
                    'waiter_id' => $cartData['waiter_id'] ?? null,
                    'status' => $cartData['status'] ?? 'submitted',
                    'flag' => $cartData['flag'] ?? 1,
                    'last_activity_at' => now(), // Set initial activity time
                ]);
            }

            $activityOccurred = false;

            // ✅ Insert or update cart items
            foreach ($cartItems as $item) {
                $existingItem = CartItems::where('cart_id', $cart->id)
                    ->where('product_id', $item['product_id'])
                    ->first();

                if ($existingItem) {
                    // Update only if quantity is increased or total changed
                    $existingItem->qty += $item['qty'];
                    $existingItem->total += $item['total'];
                    $existingItem->update();
                } else {
                    // New item — create it
                    CartItems::create([
                        'cart_id' => $cart->id,
                        'image' => $item['image'] ?? null,
                        'product_id' => $item['product_id'],
                        'product_name' => $item['product_name'],
                        'description' => $item['description'] ?? null,
                        'qty' => $item['qty'],
                        'price' => $item['price'],
                        'unit_id' => $item['unit_id'],
                        'total' => $item['total'],
                    ]);
                }

                $activityOccurred = true;
            }

            // ✅ Update last_activity_at if anything changed
            if ($activityOccurred) {
                $cart->last_activity_at = now();
                $cart->touch(); // Also updates updated_at
                $cart->save();
            }

            // ✅ Calculate order total
            $orderTotal = array_sum(array_column($cartItems, 'total'));

            $tableData = Table::find($tableId);

            // Mark table as occupied
            if ($tableData && $tableData->flag != 'occupied') {
                $tableData->update(['flag' => 'occupied']);
            }

            $locationId = optional($tableData->selling)->location_id;

            if (!$locationId) {
                return response()->json([
                    'success' => false,
                    'message' => 'Table not linked to a valid selling location.'
                ], 400);
            }

            // ✅ Check for existing sale
            $sale = Sale::where('waiter_id', 0)
                ->where('table_id', $tableId)
                ->where('status', 'credit')
                ->where('due', '>', 0)
                ->first();

            if ($sale) {
                $sale->total += $orderTotal;
                $sale->net += $orderTotal;
                $sale->due += $orderTotal;
                $sale->update();
            } else {
                $sale = Sale::create([
                    'date' => now()->toDateString(),
                    'user_id' => auth()->check() ? auth()->id() : 1,
                    'waiter_id' => 0,
                    'table_id' => $tableId,
                    'total' => $orderTotal,
                    'net' => $orderTotal,
                    'due' => $orderTotal,
                    'selling_id' => $tableData->selling_id,
                    'paid' => 0,
                    'status' => 'credit',
                ]);
            }

            // ✅ Process each cart item for sale & stock deduction
            foreach ($cartItems as $itemData) {
                $productId = $itemData['product_id'];
                $qty = $itemData['qty'];
                $price = $itemData['price'];
                $total = $itemData['total'];
                $unitId = $itemData['unit_id'];

                // Update sold items
                $soldItem = SoldItem::where('sale_id', $sale->id)
                    ->where('product_id', $productId)
                    ->first();

                if ($soldItem) {
                    $soldItem->qty += $qty;
                    $soldItem->amount += $total;
                    $soldItem->update();
                } else {
                    SoldItem::create([
                        'sale_id' => $sale->id,
                        'product_id' => $productId,
                        'unit_id' => $unitId ?? 1,
                        'cont' => 1,
                        'price' => $price,
                        'qty' => $qty,
                        'amount' => $total,
                    ]);
                }

                // ✅ Deduct stock or ingredient
                if (MenuItems::where('product_id', $productId)->exists()) {
                    $menuItems = MenuItems::where('product_id', $productId)->get();
                    foreach ($menuItems as $menuItem) {
                        $ingredient = Ingredient::find($menuItem->ingredient_id);
                        if ($ingredient) {
                            $ingredient->qty -= ($qty * $menuItem->qty);
                            $ingredient->update();
                        }
                    }
                } else {
                    recordStockMovement(
                        now()->toDateString(),
                        auth()->check() ? auth()->id() : 1,
                        $productId,
                        $locationId,
                        'Sale',
                        'Out',
                        $qty
                    );
                }
            }

            DB::commit();

            // Optional: trigger new order event/notification
            // event(new NewOrderPlaced('A new order has been placed!'));

            return response()->json([
                'success' => true,
                'message' => 'Order received and cart updated successfully.',
                'sale_id' => $sale->id,
                'cart_id' => $cart->id,
                'table' => $tableId
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Order Receive Error: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Server error while processing the order.',
                'error' => $e->getMessage()
            ], 500);
        }
    }


    public function getTables(Request $request)
    {
        $tables = Table::with('selling')->get();
        $waiters = Waiter::with('selling')->get();

        return response()->json([
            'success' => true,
            'tables' => $tables->map(function ($table) {
                return [
                    'id' => $table->id,
                    'name' => $table->name,
                    'selling_id' => $table->selling_id,
                ];
            }),
            'waiters' => $waiters->map(function ($waiter) {
                return [
                    'id' => $waiter->id,
                    'name' => $waiter->name,
                    'selling_id' => $waiter->selling_id, // Optional: adjust based on your model
                ];
            }),
        ]);
    }
}
