<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Booking;
use App\Models\Room;
use App\Models\Transaction;
use App\Models\Account;
use App\Models\Sale;
use App\Models\SoldItem;
use Carbon\Carbon;
use Exception;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class BookingController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        DB::beginTransaction();

        try {
            # Get the available rooms based on the search criteria.
            $rooms = Room::where([
                ['status', 1],
                ['block', 1],
            ])->with("price")
                ->orderBy("id", "ASC")
                ->get();
            // Get all accounts
            $accounts = Account::all();
            // Get the number of free rooms
            $roomCount = count($rooms);

            DB::commit();

            return view("booking", compact("rooms", "roomCount", "accounts"));
        } catch (Exception $e) {
            DB::rollBack();
            Log::error($e->getMessage());

            return back()->withErrors(["error" => "Something went wrong"]);
        }
    }

    public function customerCheckout()
    {
        try {
            DB::beginTransaction();

            # From the booking table get all customers checking out today and pass the to the view.
            $currentDate = date("Y-m-d");
            $customers = Booking::where("out", $currentDate)->where("status", "1")->get();

            DB::commit();

            return view("customer-checking-out", compact("customers"));
        } catch (Exception $e) {
            DB::rollBack();

            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function bookingDeposit(Request $request, $id)
    {
        try {
            DB::beginTransaction();

            # Decrement booking due & increment paid.
            $booking = Booking::findOrFail($id);
            $paidAmount = $request->paid;

            if ($paidAmount > 0) {
                $booking->paid += $paidAmount;
                $booking->due -= $paidAmount;
                $booking->update();

                # Get the account.
                $account = Account::findOrFail($request->account);

                # Record cash movement in the selected account.
                $transaction = new Transaction([
                    'date' => now()->format('Y-m-d'),
                    'reason' => "Room deposit",
                    'status' => "In",
                    'amount' => $paidAmount,
                    'before' => $account->balance,
                    'after' => $account->balance + $paidAmount,
                    'account_id' => $request->account,
                    'user_id' => auth()->user()->id,
                    'location_id' => $account->selling_location_id,
                ]);
                $transaction->save();

                # Decrement sales due and update status to cash if the debt is cleared.
                $sales = Sale::where("room_id", $booking->room_id)
                    ->orderBy("id", "ASC")
                    ->get();

                foreach ($sales as $sale) {
                    if ($paidAmount > 0 && $sale->due > 0) {
                        if ($sale->due >= $paidAmount) {
                            $sale->paid += $paidAmount;
                            $sale->status = "cash";
                            $sale->due -= $paidAmount;
                            $paidAmount = 0;
                        } else {
                            $sale->paid += $sale->due;
                            $paidAmount -= $sale->due;
                            $sale->due = 0;
                        }
                        $sale->update();
                    } else {
                        // If the paid amount becomes zero or the sales due is zero, break the loop.
                        break;
                    }
                }
            }

            DB::commit();

            return redirect()->back()->with("success", "Deposited successfully.");
        } catch (Exception $e) {
            DB::rollBack();
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function customerBookings()
    {
        DB::beginTransaction();

        try {
            # From the booking table get all today's booking and pass it to the view.
            $date =  date('Y-m-d');
            $bookings = Booking::where("status", "2")->where("in", $date)->get();
            return view("customer-bookings", compact("bookings"));
        } catch (Exception $e) {
            DB::rollBack();

            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
        DB::commit();
    }

    public function insideCustomers($status)
    {
        DB::beginTransaction();

        try {
            # From the booking table get all in house guests and pass them to the view.
            $accounts = Account::all();
            $insideCustomers = Booking::where("status", "1")->with("room", "account")->get();
            $rooms = Room::where("status", "2")
                ->orderBy("id", "ASC")
                ->get();
            $freeRooms = Room::where("status", "1")
                ->orderBy("id", "ASC")
                ->get();

        } catch (Exception $e) {
            DB::rollBack();

            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }

        DB::commit();

        return view("inside-customers", compact(
            "insideCustomers",
            "status",
            "rooms",
            "freeRooms",
            "accounts"
        ));
    }

    public function upcomingBookings()
    {
        DB::beginTransaction();

        try {
            # From the booking table get all bookings and pass then to the view.
            $accounts = Account::all();
            $bookings = Booking::where("status", "2")->with("room")->get();
        } catch (Exception $e) {
            DB::rollBack();

            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }

        DB::commit();

        return view("upcoming-bookings", compact(
            "bookings",
            "accounts"
        ));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function vat($total)
    {
        DB::beginTransaction();

        try {
            $answer = (18 / 100) * $total;
        } catch (Exception $e) {
            DB::rollBack();

            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }

        DB::commit();

        return $answer;
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function booking(Request $request, $id)
    {
        try {
            DB::beginTransaction();

            // Validate the request data
            $validated = $request->validate([
                'in' => 'required|date',
                'customer' => 'required|string',
                'phone' => 'required|string',
                'comment' => 'nullable|string',
                'price' => 'required|numeric',
                'paid' => 'required|numeric',
                'status' => 'required|integer',
                'account' => 'nullable|exists:accounts,id',
            ]);

            // Inserting booking.
            $booking = new Booking();
            $booking->user_id = auth()->user()->id;
            $booking->in_date = $validated['in'];
            $booking->days = 1;
            $booking->customer = ucwords($validated['customer']);
            $booking->phone = $validated['phone'];
            $booking->comment = $validated['comment'];
            $booking->room_id = $id;
            $booking->total = $validated['price'];
            $booking->paid = $validated['paid'];
            $booking->due = $validated['price'] - $validated['paid'];

            if ($validated['paid'] > 0 && $request->filled('account')) {
                $booking->account_id = $validated['account'];
            }

            $booking->status = $validated['status'];
            $booking->total_bill = 0;
            $booking->save();

            // Update room status.
            $room = Room::findOrFail($id);
            $room->status = 2;
            $room->save();

            // Insert sales record.
            $sale = new Sale();
            $sale->date = $validated['in'];
            $sale->user_id = auth()->user()->id;
            $sale->room_id = $id;
            $sale->total = $validated['price'];
            $sale->paid = $validated['paid'];
            $sale->due = $validated['price'] - $validated['paid'];
            $sale->net = $validated['price'] - $validated['paid'];
            $sale->status = $booking->due > 0 ? "credit" : "cash";

            if ($validated['paid'] > 0 && $request->filled('account')) {
                $sale->account_id = $validated['account'];
            }
            $sale->save();

            // Inserting sold items.
            $items = new SoldItem();
            $items->sale_id = $sale->id;
            $items->unit_id = "Days";
            $items->cont = 1;
            $items->price = $validated['price'];
            $items->qty = 1;
            $items->amount = $validated['price'];
            $items->save();

            // If the amount paid is greater than 0, record the cash movement.
            if ($validated['paid'] > 0 && $request->filled('account')) {
                $account = Account::findOrFail($validated['account']);

                $transaction = Transaction::create([
                    'date' => now()->format('Y-m-d'),
                    'reason' => ($validated['status'] === 1) ? "Checkin payment" : "Booking payment",
                    'status' => "in",
                    'amount' => $validated['paid'],
                    'before' => $account->balance,
                    'after' => $account->balance + $validated['paid'],
                    'account_id' => $validated['account'],
                    'user_id' => auth()->user()->id,
                    'location_id' => $account->selling_location_id,
                ]);

                // Update account balance.
                $account->balance += $validated['paid'];
                $account->save();
            }

            DB::commit();

            return back()->with("success", $booking->status == 1 ? "Successfully checked in!!" : "Successfully booked!!");

        } catch (Exception $e) {
            DB::rollBack();
            Log::error("Booking transaction failed: " . $e->getMessage());
            return response()->json(['error' => 'Something went wrong'], 500);
        }

    }


    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        DB::beginTransaction();

        try {
            $booking = Booking::findOrFail($id);
            $room = Room::find($booking->room_id);

            $startDate = Carbon::parse($request->input('in'));
            $endDate = Carbon::parse($request->input('out'));
            $days = $endDate->diffInDays($startDate);

            $sale = Sale::where("room_id", $booking->room_id)->latest()->first();

            if ($days !== $booking->days) {
                $totalPrice = $room->price * $days;
                $netAmount = $totalPrice - $sale->paid;

                $sale->update([
                    'total' => $totalPrice,
                    'net' => $netAmount,
                    'due' => $netAmount,
                ]);

                $soldData = SoldItem::where('sale_id', $sale->id)->first();
                $soldData->update([
                    'qty' => $days,
                    'amount' => $totalPrice,
                ]);
            }

            $booking->update([
                'in_date' => $request->input('in'),
                'out_date' => $request->input('out'),
                'days' => $days,
                'customer' => ucwords($request->input('customer')),
                'phone' => $request->input('phone'),
                'comment' => $request->input('comment'),
                'total' => $request->input('price') * $days,
                'due' => $request->input('price') * $days - $request->input('paid'),
            ]);

            DB::commit();

            return back()->with("success", "Booking updated.");
        } catch (\Exception $e) {
            DB::rollback();
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return back()->with("error", "Something went wrong");
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        try {
            DB::beginTransaction();

            # Get the selected booking.
            $booking = Booking::find($id);
            # Get the selected room.
            $room = Room::find($booking->room_id);
            # Release the room $ update.
            $room->status = "free";
            $room->update();
            # Get the sale for that room.
            $sale = Sale::where("room_id", $booking->room_id)->latest()->first();
            # From the sold item get the item with the above sale id & delete it.
            SoldItem::where("sale_id", $sale->id)->delete();
            # Delete the sale.
            $sale->delete();
            #  If the paid amount is greater than 0.
            if ($booking->account_id !== null) {
                # Get the selected account.
                $account  = Account::find($booking->account_id);

                # Record the cash movement in above account.
                $transaction = new Transaction([
                    'date' => date('Y-m-d'),
                    'reason' => "Check-in deleted",
                    'status' => "in",
                    'amount' => $booking->paid,
                    'before' => $account->balance,
                    'after' => $account->balance - $booking->paid,
                    'account_id' => $booking->account_id,
                    'user_id' => auth()->user()->id,
                    'location_id' => $account->selling_location_id,
                ]);
                $transaction->save();
                # Update the account balance.
                $account->balance -=  $booking->paid;
                $account->update();
            }
            # Finally delete the booking.
            $booking->delete();

            DB::commit();

            return back()->with("success", "Successfully cancelled.");
        } catch (Exception $e) {
            DB::rollback();
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function bookingCheckin($id)
    {
        try {
            DB::beginTransaction();

            # Get the booking with the passed id then update status column.
            $booking = Booking::findOrFail($id);
            $booking->status = "1";
            $booking->update();

            DB::commit();

            return back()->with("success", "Successfully checked-in.");
        } catch (Exception $e) {
            DB::rollback();
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function checkoutView($room)
    {
        DB::beginTransaction();

        try {
            $items = DB::table("bills")
                ->join("bill_items", "bill_items.bill_id", "=", "bills.id")
                ->where("bills.room", $room)->get();

            $view = view("check out", compact("items"))->render();

            DB::commit();

            return $view;
        } catch (Exception $e) {
            DB::rollback();

            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function checkout(Request $request, $id)
    {
        DB::beginTransaction();

        try {
            // Get the booking with the passed id then update its details.
            $booking = Booking::findOrFail($id);

            // Get the room then mark it as free.
            $room = Room::find($booking->room_id);
            $room->status = "1";
            $room->save();

            // Get the account.
            $account = Account::findOrFail($request->account);

            // Record cash movement in the selected account.
            $transaction = new Transaction([
                'date' => now()->format('Y-m-d'),
                'reason' => "Check-out payment",
                'status' => "In",
                'amount' => $request->amount,
                'before' => $account->balance,
                'after' => $account->balance + $request->amount,
                'account_id' => $request->account,
                'user_id' => auth()->user()->id,
                'location_id' => $account->selling_location_id ? $account->selling_location_id : null,
            ]);
            $transaction->save();

            // Update account balance.
            $account->increment('balance', $request->amount);

            // Update sales status to cash if it was on credit.
            $sale = Sale::where("room_id", $booking->room_id)->latest()->first();
            if ($sale->status === "credit") {
                $sale->update([
                    'status' => "cash",
                    'paid' => $sale->net,
                    'due' => 0,
                    'account_id' => $request->account,
                ]);
            }

            // Delete booking.
            $booking->delete();

            DB::commit();

            return redirect()->back()->with("success", "Customer checked out successfully.");
        } catch (Exception $e) {
            DB::rollback();

            // Log the error.
            Log::error($e->getMessage());
            // Return a custom error response.
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }
}
