<?php

namespace App\Http\Controllers;

use App\Imports\CustomerImport;
use App\Models\Account;
use Illuminate\Http\Request;
use App\Models\Customer;
use App\Models\Debts;
use App\Models\Selling;
use App\Models\Transaction;
use Exception;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Maatwebsite\Excel\Facades\Excel;

class CustomerController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        try {
            # Get location details
            $locationDetails = processLocationDetails();
            $sellingCount = $locationDetails['sellingCount'];
            $sellings = $locationDetails['sellings'];
            $accounts = $locationDetails['accounts'];
            $customers = $locationDetails['customers'];

            return view("customer", compact("customers", "accounts", "sellingCount", "sellings"));
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function uploadCustomerExcel(Request $request)
    {
        try {
            Excel::import(new CustomerImport, $request->file);

            return back()->with("success", "Excel uploaded!!");
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        // Begin the database transaction
        DB::beginTransaction();

        try {
            $request->validate([
                'name' => [
                    'required',
                    function ($attribute, $value, $fail) use ($request) {
                        if (Customer::where('name', $value)->where('selling_id', $request->location)->exists()) {
                            $fail('The customer exists in this location.');
                        }
                    },
                ],
            ]);

            $customer = new customer();
            $customer->name = ucwords($request->name);
            $customer->address = ucwords($request->address);
            $customer->phone = $request->phone;
            $customer->email = $request->email;
            $customer->limit = $request->limit;
            $customer->advance = $request->advance;
            $customer->opening_balance = $request->opening;
            $customer->selling_id = $request->location;
            $customer->save();

            if ($request->has('opening')) {
                $debt = new Debts();
                $debt->date = date('Y-m-d');
                $debt->customer_id = $customer->id;
                $debt->opening_balance = $request->opening;
                $debt->due = $request->opening;
                $debt->selling_id = $request->location;
                $debt->save();
            }

            // Commit the database transaction
            DB::commit();

            return back()->with("success", "Successfully added!!");
        } catch (Exception $e) {
            // An error occurred, rollback the transaction
            DB::rollBack();
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function advancePayment(Request $request, $id)
    {
        try {
            $customer = Customer::find($id);
            $customer->advance = $request->amount;
            $customer->description = $request->description;
            $customer->update();

            $account = Account::where("name", "LIKE", "%$request->account%")
                ->where("selling_id", $request->location)
                ->first();

            $transaction = new Transaction();

            $transaction->date = date('Y-m-d');
            $transaction->user_id = auth()->user()->id;
            $transaction->reason =  "Supplier advance ($request->name)";
            $transaction->status = "Out";
            $transaction->amount = $request->amount;
            $transaction->before = $account->balance;
            $transaction->after = $account->balance - $request->amount;
            $transaction->account_id  = $account->id;
            $transaction->selling_id = $request->location;
            $transaction->save();

            $account->balance -=  $request->amount;
            $account->update();

            return back()->with("success", "Successfully posted.");
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function getCustomer(Request $request)
    {
        try {
            $query = $request->get('query');
            $response = Customer::where('name', 'LIKE', "%$query%")
                ->take(10) // Limit the number of results to 10
                ->get();

            return response()->json($response);
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function creditLimit(Request $request) {}

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {

        // Begin the database transaction
        DB::beginTransaction();

        try {
            # Find the customer
            $customer = Customer::find($id);
            $sellingId = $request->location;

            # If the selling location changes
            if ($sellingId != $customer->selling_id) {
                Debts::where('customer_id', $id)
                    ->where('selling_id', $customer->selling_id)
                    ->update(['selling_id' => $sellingId]);
            }

            # Check if there is any change in oppening balance.
            if ($request->opening != $customer->opening_balance) {
                $customerDebtData = Debts::where('customer_id', $id)
                    ->whereNotNull('opening_balance')
                    ->where('selling_id', $sellingId)
                    ->first();

                if ($customerDebtData) {
                    # Update opening balance.
                    $customerDebtData->update(['opening_balance' => $request->opening]);

                    # Recaluculate the entire customer debts
                    updateCustomerDebts($sellingId, $id);
                }
            }

            # Update customer details.
            $customer->name = ucwords($request->name);
            $customer->address = ucwords($request->address);
            $customer->phone = $request->phone;
            $customer->email = $request->email;
            $customer->limit = $request->limit;
            $customer->advance = $request->advance;
            $customer->description = $request->description;
            $customer->opening_balance = $request->opening;
            $customer->selling_id = $sellingId;
            $customer->update();

            // Commit the database transaction
            DB::commit();

            return back()->with("success", "Successfully updated!!");
        } catch (Exception $e) {
            // An error occurred, rollback the transaction
            DB::rollBack();
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }


    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function bulkDelete(Request $request)
    {
        try {
            $ids = $request->ids;

            $results = "";

            foreach ($ids as $id) {
                $results = Customer::find($id)->delete();
            }
            return response()->json($results);
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function destroy($id)
    {
        try {
            Customer::find($id)->delete();

            return back()->with("success", "Successfully deleted.");
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }
}
