<?php

namespace App\Http\Controllers;

use App\Imports\IngredientImport;
use App\Models\Ingredient;
use App\Models\MenuItems;
use App\Models\Product;
use App\Models\ProductCategory;
use App\Models\Stock;
use App\Models\Storage;
use App\Models\Unit;
use App\Models\UnitAssigned;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Maatwebsite\Excel\Facades\Excel;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class IngredientsController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        # Get location details
        $locationDetails = processLocationDetails();
        $sellingCount = $locationDetails['sellingCount'];
        $sellings = $locationDetails['sellings'];

        $units = Unit::all();

        # Initiate variable
        $categoryIds = [];

        $categories = ProductCategory::where('id', 3)
            ->orWhere('parent_id', 3)->get();

        $ingredients = Product::whereIn('category_id', $categories->pluck('id')->toArray())
            ->with('stock', 'unit.unit', 'category', 'location')->get();

        return view("ingredients", compact("ingredients", "units", "sellings", "categories"));
    }


    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {

        DB::beginTransaction(); // Start a transaction
        try {
            # Abort if the product already exists.
            if (Product::where('product', ucwords($request->product))->where('location_id', $request->location)->exists()) {
                return back()->with('error', "{$request->product} already exists in {$request->location}");
            }

            // Initialize the image file name
            $fileName = '';

            # Image codes
            if (request()->image) {
                if ($request->hasFile('image')) {
                    $image = $request->file('image');
                    $fileName = uniqid('image_') . '.' . $image->getClientOriginalExtension();
                    // Store the image in the public disk
                    $image->storeAs('public/images', $fileName);
                }
            }

            # If the product is a normal product...insert product details
            $product = new Product();
            $product->barcode = $request->barcode != null ? $request->barcode : str_pad(rand(0, 9999999999999), 13, '0', STR_PAD_LEFT);
            $product->product = ucwords($request->name);
            $product->image = $fileName ?? ''; // If there is a file name the insert it if not insert an empty string.
            $product->category_id = $request->category ?? 3; // Default to category ID 3 (Ingredients) if not provided
            $product->buying = $request->buying;
            $product->location_id = $request->location;
            $product->save();

            # Insert unit_assigned details
            $unit = new UnitAssigned();
            $unit->product_id = $product->id;
            $unit->unit_id = $request->unit;
            $unit->unit_cont = 1;
            $unit->selling = 0;
            $unit->location_id = $request->location;
            $unit->save();

            recordStockMovement(
                date('Y-m-d'),
                Auth::id(),
                $product->id,
                $request->location,
                "Registered",
                "In",
                $request->qty
            );

            // Commit the transaction if everything is successful
            DB::commit();

            return back()->with("success", "Ingredient successfully Registered!!");
        } catch (Exception $e) {
            // Rollback the transaction in case of an error
            DB::rollback();
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function getIngredients(Request $request)
    {
        try {
            $query = $request->get('data');

            $response = Ingredient::where('name', 'like', '%' . $query . '%')
                ->with('unit')
                ->take(15)
                ->get();

            return response()->json($response);
        } catch (\Exception $e) {
            Log::error('Error fetching ingredients: ' . $e->getMessage());
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }


    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function uploadIngredients(Request $request)
    {
        try {
            Excel::import(new IngredientImport, $request->file);
            return back()->with("success", "Excel successfully uploaded.");
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */

    public function update(Request $request, $id)
    {
        DB::beginTransaction();
        try {
            // Find the product
            $product = Product::findOrFail($id);

            /** ---------------------------
             * Handle Image Update
             * ---------------------------
             */
            $fileName = $product->image; // keep old image by default
            if ($request->hasFile('image')) {
                // Delete old image if exists
                if ($fileName && Storage::exists('public/images/' . $fileName)) {
                    Storage::delete('public/images/' . $fileName);
                }

                $image = $request->file('image');
                $fileName = uniqid('image_') . '.' . $image->getClientOriginalExtension();
                $image->storeAs('public/images', $fileName);
            }

            /** ---------------------------
             * Update Product Details
             * ---------------------------
             */
            $product->product     = ucwords($request->name);
            $product->image       = $fileName;
            $product->category_id = $request->category;
            $product->buying      = $request->buying;
            $product->location_id = $request->location;
            $product->save();

            /** ---------------------------
             * Update Unit Assignment
             * ---------------------------
             */
            $unit = UnitAssigned::where('product_id', $product->id)->first();
            if ($unit) {
                $unit->unit_id     = $request->unit;
                $unit->location_id = $request->location;
                $unit->save();
            }

            /** ---------------------------
             * Update Stock Details
             * ---------------------------
             */
            $stock = Stock::where('product_id', $product->id)->first();
            if ($stock) {
                $oldQty = $stock->qty;
                $newQty = (int) $request->qty;

                $stock->location_id = $request->location;
                $stock->unit_id     = $request->unit;
                $stock->min_cont    = $request->mincont;
                $stock->min_qty     = $request->minqty;
                $stock->exp         = $request->exp;
                $stock->exp_not     = $request->expnot;
                $stock->save();

                // Only record stock movement if quantity has changed
                if ($newQty !== $oldQty) {
                    $qtyDifference = $newQty - $oldQty;
                    $movementType  = $qtyDifference > 0 ? "In" : "Out";
                    $movementQty   = abs($qtyDifference);

                    recordStockMovement(
                        date('Y-m-d'),
                        Auth::id(),
                        $product->id,
                        $request->location,
                        "Update",
                        $movementType,
                        $movementQty
                    );
                }
            }

            DB::commit();
            return back()->with("success", "Product successfully updated!");
        } catch (Exception $e) {
            DB::rollBack();
            Log::error($e->getMessage());
            return back()->with('error', 'Something went wrong while updating.');
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */

    public function destroy($id)
    {
        DB::beginTransaction();
        try {
            // Find the product
            $product = Product::findOrFail($id);

            // Get current stock for logging movement
            $stock = Stock::where('product_id', $product->id)->first();

            if ($stock && $stock->qty > 0) {
                // Record stock movement as "Out" before deletion
                recordStockMovement(
                    date('Y-m-d'),
                    Auth::id(),
                    $product->id,
                    $stock->location_id,
                    "Delete",
                    "Out",
                    $stock->qty
                );
            }

            // Delete product image if exists
            if ($product->image && Storage::exists('public/images/' . $product->image)) {
                Storage::delete('public/images/' . $product->image);
            }

            // Delete stock
            Stock::where('product_id', $product->id)->delete();

            // Delete unit assignments
            UnitAssigned::where('product_id', $product->id)->delete();

            // Finally delete product
            $product->delete();

            DB::commit();
            return back()->with('success', 'Deleted Successfully!');
        } catch (Exception $e) {
            DB::rollBack();
            Log::error($e->getMessage());
            return back()->with('error', 'Something went wrong while deleting the product.');
        }
    }


    public function bulkDelete(Request $request)
    {
        DB::beginTransaction(); // Start a database transaction

        try {
            $ids = $request->ids;
            $results = "";

            foreach ($ids as $id) {
                Ingredient::find($id)->delete();
                MenuItems::where("ingredient_id", $id)->delete();
            }

            DB::commit(); // Commit the transaction

            return response()->json($results);
        } catch (\Exception $e) {
            DB::rollBack(); // Rollback the transaction in case of an error
            Log::error($e->getMessage());
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function deleteMenuItem($id)
    {
        DB::beginTransaction(); // Start a database transaction

        try {
            MenuItems::find($id)->delete();
            DB::commit(); // Commit the transaction

            return redirect()->back()->with("success", "Successfully deleted");
        } catch (\Exception $e) {
            DB::rollBack(); // Rollback the transaction in case of an error
            Log::error($e->getMessage());
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }
}
