<?php

namespace App\Http\Controllers;

use App\Models\Account;
use App\Models\Sale;
use App\Models\Booking;
use App\Models\BusinessProfile;
use App\Models\Cart;
use App\Models\CartItems;
use App\Models\Ingredient;
use App\Models\Location;
use App\Models\MenuItems;
use App\Models\Product;
use App\Models\ProductCategory;
use App\Models\Selling;
use App\Models\SoldItem;
use App\Models\Stock;
use App\Models\StockMovement;
use App\Models\Table;
use App\Models\Transaction;
use App\Models\Unit;
use App\Models\Waiter;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Session;

class PlaceOderController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index($status)
    {
        try {
            # Get location details
            $locationDetails = processLocationDetails();
            $sellingCount = $locationDetails['sellingCount'];
            $sellings = $locationDetails['sellings'];
            $waiters = Waiter::all();

            return view("place-oder", compact(
                "status",
                "sellings",
                "waiters"
            ));
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function order()
    {
        try {
            $role = auth()->user()->role;
            if ($role == "Counter") {
                # Get all categories
                $categories = ProductCategory::where("name", "!=", null)
                    ->Where("name", "!=", "Service")
                    ->Where("name", "!=", "Ingredient")
                    ->get();

                # Get all categories
                $tables = Table::all();
                # Get all selling locations.
                $locations = Location::where("type", "selling")->get();
                // Retrieve data from the session
                $waiter_id = Session::get('waiterId');

                return view("order", compact("categories", "tables", "locations", "waiter_id"));
            } else {
                return view("order");
            }
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function cancelOrderView()
    {
        try {
            # Get location details
            $locationDetails = processLocationDetails();
            $sellings = $locationDetails['sellings'];
            $waiters = $locationDetails['waiters'];
            $tables = $locationDetails['tables'];

            $categories = ProductCategory::where("name", "!=", null)
                ->Where("name", "!=", "Service")
                ->Where("name", "!=", "Ingredient")
                ->get();

            return view("cancel-oder", compact(
                "sellings",
                "categories",
                "waiters",
                "tables"
            ));
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function payBillView()
    {
        try {
            # Get location details
            $locationDetails = processLocationDetails();
            $sellings = $locationDetails['sellings'];
            $accounts = $locationDetails['accounts'];
            $waiters = $locationDetails['waiters'];
            $tables = $locationDetails['tables'];

            return view("pay-bills", compact(
                "sellings",
                "tables",
                "waiters",
                "accounts",
            ));
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function waitersBillView()
    {
        try {
            $results = DB::table("sales")
                ->join("sold_items", "sold_items.sale_id", "=", "sales.id")
                ->where("sales.status", "=", "credit")
                ->where("sales.due", ">", 0)
                ->get();

            # Get location details
            $locationDetails = processLocationDetails();
            $waiters = $locationDetails['waiters'];
            $tables = $locationDetails['tables'];

            $billId = 0;

            return view("waiters-bill", compact("results", "tables", "waiters", "billId"));
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function validateWaitersDetails(Request $request)
    {
        try {
            if (Session::has('waiterId')) { # The session key exists and is not empty
                // Clear all data from the session
                Session::flush();
            }

            # From waiters table find the selected waiter.
            $waiterData = Waiter::find($request->waiter);
            $waiterId = $waiterData->id;
            $sellingId = $waiterData->selling_id;
            $password = $request->password;

            # Get all categories
            $categories = ProductCategory::where("name", "!=", null)
                ->Where("name", "!=", "Service")
                ->Where("name", "!=", "Ingredient")
                ->get();

            # Get waiters location table
            $tables = Table::where('selling_id', $sellingId)->get();

            # Don't grant access if the password doesn't match.
            if ($waiterData->password != $password) {
                return redirect()->back()->with('error', 'Wrong password');
            } else {  # Grant access if the password match.
                if (auth()->user()->role == "Counter") {
                    // Store data in the session
                    Session::put('waiterId', $waiterId);
                }

                // Get the 5 most recent open carts with their items
                $recentOrders = Cart::with('cartItems')
                    ->where('flag', 1)
                    ->get();

                // Count how many carts were retrieved
                $recentOrdersCount = $recentOrders->count();

                return view("order", compact(
                    "waiterId",
                    "password",
                    "tables",
                    "categories",
                    "recentOrders",
                    "recentOrdersCount",
                ));
            }
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function RecentOrders()
    {
        $recentOrders = Cart::with('cartItems')
            ->where('flag', 1) // or status = 'submitted', depending on your logic
            ->whereNotNull('last_activity_at')
            ->orderByDesc('last_activity_at')
            ->get();

        return response()->json([
            'count' => $recentOrders->count(),
            'orders' => $recentOrders,
        ]);
    }

    public function renderModal($cartId)
    {
        $cart = Cart::with('cartItems', 'table')->findOrFail($cartId);
        return view('partials.order_modal', compact('cart'))->render();
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */

    public function waitersOder(Request $request)
    {
        try {
            # Get the entered details
            $location = $request->get('location');
            $table = $request->get('table');
            $waiter = $request->get('waiter');
            $category = $request->get('category');
            $query = $request->get('query');

            # From waiters order get the queried items
            $results = DB::table("sales")
                ->join("sold_items", "sales.id", "=", "sold_items.sale_id")
                ->where("sales.location_id", $location)
                ->where("sales.table_id", $table)
                ->where("sales.waiter_id", $waiter)
                ->where("sales.status", "credit")
                ->where("sold_items.product", "like", "%$query%")
                ->get();

            # In the products table check the category of the waiters order.
            // write your codes here.
            return response()->json($results);
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function getBill(Request $request)
    {
        try {
            $sellingId = $request->get('sellingId');
            $tableId = $request->get('tableId');
            $waiterId = $request->get('waiterId');

            $response = Sale::where("status", "credit")
                ->where("due", ">", 0)
                ->orderBy('created_at', 'asc')
                ->when($sellingId !== "all", function ($query) use ($sellingId) {
                    return $query->where("selling_id", $sellingId);
                })
                ->when($tableId !== "all", function ($query) use ($tableId) {
                    return $query->where("table_id", $tableId);
                })
                ->when($waiterId !== "all", function ($query) use ($waiterId) {
                    return $query->where("waiter_id", $waiterId);
                })
                ->sum("due");

            return response()->json($response);
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function getWaiterBill(Request $request)
    {
        try {
            $tableId = $request->get('tableId');
            $waiterId = $request->get('waiterId');

            $sellingId = Table::find($tableId)->value('selling_id');

            $response = Sale::where("status", "credit")
                ->where("due", ">", 0)
                ->where("waiter_id", $waiterId)
                ->where("table_id", $tableId)
                ->where("selling_id", $sellingId)
                ->with('soldItems.product', 'soldItems.unit', 'selling.location')
                ->orderBy('created_at', 'asc')
                ->first();

            return response()->json($response);
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function clearBill(Request $request)
    {
        DB::beginTransaction();

        try {
            $date = $request->get("date");
            $sellingId = $request->get("area");
            $tableId = $request->get("table");
            $waiterId = $request->get("waiter");
            $paid = (int)$request->get("paid", 0);
            $accountId = $request->get("account");

            $salesQuery = Sale::where("date", $date)
                ->where("status", "credit");

            if ($sellingId !== "all") {
                $salesQuery->where("selling_id", $sellingId);
            }
            if ($tableId !== "all") {
                $salesQuery->where("table_id", $tableId);
            }
            if ($waiterId !== "all") {
                $salesQuery->where("waiter_id", $waiterId);
            }

            $sales = $salesQuery->get();

            if ($sales->isEmpty()) {
                return back()->with("info", "No pending bills found for the selected criteria.");
            }

            $remainingPaid = $paid;
            $qrTablesToNotify = [];

            foreach ($sales as $sale) {
                if ($sale->due <= 0) {
                    continue;
                }

                if ($sellingId === "all" && $tableId === "all" && $waiterId === "all") {
                    // Clear all completely
                    $sale->paid = $sale->net;
                    $sale->due = 0;
                } else {
                    // Apply partial or full amount
                    $applyAmount = min($remainingPaid, $sale->due);
                    $sale->paid += $applyAmount;
                    $sale->due -= $applyAmount;
                    $remainingPaid -= $applyAmount;
                }

                $sale->account_id = $accountId;

                if ($sale->due <= 0) {
                    $sale->status = "cash";
                    $sale->paid = $sale->net;
                    $sale->due = 0;

                    // ✅ Immediately mark table as free
                    Table::where('id', $sale->table_id)->update(['flag' => 'free']);

                    // Track QR tables for notification
                    if ((int)$sale->waiter_id === 0 && !in_array($sale->table_id, $qrTablesToNotify)) {
                        $qrTablesToNotify[] = $sale->table_id;
                    }
                }

                $sale->save();

                if ($remainingPaid <= 0 && $paid > 0) {
                    break;
                }
            }

            // Record transaction
            if ($paid > 0) {
                $account = Account::findOrFail($accountId);

                Transaction::create([
                    'date' => $date,
                    'user_id' => auth()->id(),
                    'reason' => "Sale paid",
                    'status' => "In",
                    'amount' => $paid,
                    'before' => $account->balance,
                    'after' => $account->balance + $paid,
                    'account_id' => $accountId,
                    'selling_id' => $account->selling_id,
                ]);

                $account->balance += $paid;
                $account->save();
            }

            // Notify QR system
            $qrTablesToNotify = array_unique($qrTablesToNotify);

            if (!empty($qrTablesToNotify)) {
                try {
                    $response = Http::timeout(5)->post('https://thecaveqr.dukamax.online/api/qr-order-cleared', [
                        'table_ids' => $qrTablesToNotify
                    ]);

                    if (!$response->successful() || !$response->json('success')) {
                        DB::rollBack();
                        Log::error('QR system update failed: ' . $response->body());
                        return back()->with('error', 'Bill not cleared because QR system update failed.');
                    }
                } catch (\Exception $apiError) {
                    DB::rollBack();
                    Log::error("QR API call failed: " . $apiError->getMessage());
                    return back()->with('error', 'Bill not cleared because QR system update failed.');
                }
            }

            DB::commit();
            return back()->with("success", "Bill successfully cleared.");
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error("Bill clear failed: " . $e->getMessage());
            return response()->json(['error' => 'Something went wrong. Please try again.'], 500);
        }
    }

    public function printBill($sellingId, $tableId, $waiterId)
    {
        // Fetch the bill details for the specified location, table, and waiter
        $bill = Sale::where('selling_id', $sellingId)
            ->where('waiter_id', $waiterId)
            ->where('table_id', $tableId)
            ->where('status', 'credit')
            ->with('waiter', 'table', 'selling.location')
            ->first();

        // Fetch the sold items associated with the bill
        $billItems = SoldItem::where('sale_id', $bill->id)
            ->with('product', 'unit')->get();

        // Initialize an empty array to store the order details
        $orderArray = [
            "sellingId" => [],
            "id" => [],
            "product" => [],
            "unit" => [],
            "price" => [],
            "qty" => [],
            "amount" => [],
        ];

        // Loop through the sold items and populate the order array
        foreach ($billItems as $item) {
            $sellingId = $bill->selling_id;
            $id = $item->product_id;
            $product = $item->product->product; // Assuming there is a relation with Product model
            $unit = $item->unit->name; // Assuming the unit is available in sold item
            $qty = $item->qty;
            $price = $item->price;
            $amount = $item->amount;

            // Find if the combination of "location" and "id" already exists in the array
            $index = array_search([$sellingId, $id], array_map(null, $orderArray["sellingId"], $orderArray["id"]));

            if ($index !== false) {
                // If found, increment the quantity and amount for that product
                $orderArray["qty"][$index] += $qty;
                $orderArray["amount"][$index] += $amount;
            } else {
                // Otherwise, add the new entry to the order array
                $orderArray["sellingId"][] = $sellingId;
                $orderArray["id"][] = $id;
                $orderArray["product"][] = $product;
                $orderArray["unit"][] = $unit;
                $orderArray["price"][] = $price;
                $orderArray["qty"][] = $qty;
                $orderArray["amount"][] = $amount;
            }
        }

        // Now you can proceed to process $orderArray, e.g., for printing or further operations
        $dateTime = date("Y-m-d H:i:s");
        $status = "split";
        $role = auth()->user()->role;
        $waiterData = $bill->waiter;
        $tableData = $bill->table;
        $orderTotal = $bill->total;
        $profileData = BusinessProfile::where("selling_id", $bill->selling_id)->first();

        // Initialize the order number variable
        $orderNumber = 0;

        // Get the current date and time in HH:MM:SS format
        $currentTime = date('H:i:s'); // This will return the current time, e.g., '17:27:41'
        $currentDate = date('Y-m-d'); // This will return the current date, e.g., '2024-09-20'

        // Check if the last generated date is stored in the session
        if (session()->has('last_generated_date')) {
            // Compare the current date with the last generated date
            if ($currentDate !== session('last_generated_date')) {
                // If it's a new day, reset the order number to 0
                $orderNumber = 0;

                // Update the session with the current date
                session(['last_generated_date' => $currentDate]);
            } else {
                // If it's the same day, retrieve the existing order number from the session
                $orderNumber = session('order_number', 0);
            }
        } else {
            // If there's no last generated date in the session, initialize it
            session(['last_generated_date' => $currentDate]);
        }

        // Increment the order number by 1
        $orderNumber += 1;

        // Store the updated order number back in the session for future reference
        session(['order_number' => $orderNumber]);

        return view("print-bill", compact(
            "role",
            "dateTime",
            "profileData",
            "orderArray",
            "orderTotal",
            "waiterData",
            "tableData",
            "orderNumber",
            "status"
        ));
    }

    public function qrCodeOrders($waiterId, $password)
    {
        // Eager load cart items for all carts (optional: filter by status if you want)
        $carts = Cart::with('cartItems', 'table')->get();

        return view('qr-code-orders', compact('carts', 'waiterId', 'password'));
    }

    public function markAsSeen($cartId)
    {
        DB::beginTransaction();

        try {
            // Attempt to find the cart
            $cart = Cart::find($cartId);

            if (!$cart) {
                DB::rollBack();
                return redirect()->back()->with('error', 'Cart not found.');
            }

            // Update the flag
            $cart->update(['flag' => 2]);

            // Commit transaction
            DB::commit();

            return redirect()->back()->with('success', 'Cart flag updated successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->with('error', 'Something went wrong: ' . $e->getMessage());
        }
    }

    public function fetchNewOrders()
    {
        $recentOrders = Cart::with('cartItems', 'table')
            ->where('flag', 1)
            ->limit(5)
            ->get();

        $recentOrdersCount = $recentOrders->count();

        // Return JSON for AJAX
        return response()->json([
            'count' => $recentOrdersCount,
            'orders' => $recentOrders
        ]);
    }

    public function cancelOrder(Request $request)
    {
        DB::beginTransaction();

        try {
            $saleId = $request->input('sale-id');
            $sale   = Sale::find($saleId);

            if (!$sale) {
                return redirect()->back()->with('error', 'Sale not found.');
            }

            $tableId = $sale->table_id;
            $sellingId = Table::where('id', $tableId)->value('selling_id');
            $qrcodeSetting = BusinessProfile::where('selling_id', $sellingId)->value('qrcode');

            // ✅ FULL CANCELLATION
            if (!is_array($request->id) || count($request->id) === 0) {
                $items = SoldItem::where('sale_id', $saleId)->get();

                // Return each product to stock
                foreach ($items as $item) {
                    $locationId = Product::find($item->product_id)->location_id ?? 2; // Fallback location if missing

                    recordStockMovement(
                        date('Y-m-d'),           // date
                        auth()->user()->id,      // current user ID
                        $item->product_id,       // product ID
                        $locationId,             // location ID
                        "Cancel Sale",           // reason
                        "In",                    // status (return to stock)
                        $item->qty               // quantity returned
                    );
                }

                // If QR code integration is enabled, sync cancellation
                if ($qrcodeSetting &&  $qrcodeSetting == 1) {
                    // Sync with QR system
                    $apiResponse = Http::post('https://thecaveqr.dukamax.online/api/cancel-order', [
                        'table_id' => $tableId,
                        'items'    => [] // empty = full cancel
                    ]);

                    if ($apiResponse->successful() && ($apiResponse['success'] ?? false)) {
                        // Delete sale after stock rollback
                        SoldItem::where('sale_id', $saleId)->delete();
                        $sale->delete();
                        DB::commit();
                        return redirect()->back()->with('success', 'Bill deleted and QR system synced.');
                    } else {
                        DB::rollBack();
                        return redirect()->back()->with('error', 'Failed to sync cancellation to QR system.');
                    }
                }
            }

            // ✅ PARTIAL UPDATE
            $cleanTotal   = floatval(str_replace(',', '', $request->total));
            $sale->total  = $cleanTotal;
            $sale->net    = $cleanTotal;
            $sale->due    = $cleanTotal;
            $sale->status = 'credit';
            $sale->save();

            // Return all existing items to stock before replacing them
            $oldItems = SoldItem::where('sale_id', $saleId)->get();
            foreach ($oldItems as $oldItem) {
                $locationId = Product::find($oldItem->product_id)->location_id ?? 2;

                recordStockMovement(
                    date('Y-m-d'),
                    auth()->user()->id,
                    $oldItem->product_id,
                    $locationId,
                    "Cancel Partial Sale",
                    "In",
                    $oldItem->qty
                );
            }

            // Remove old items
            SoldItem::where('sale_id', $saleId)->delete();

            // Cache unit IDs for performance
            $unitCache = Unit::whereIn('name', $request->unit)->pluck('id', 'name');

            $newItems = [];

            foreach ($request->id as $i => $productId) {
                $unitName = $request->unit[$i];
                $unitId   = $unitCache[$unitName] ?? null;
                if (!$unitId) continue;

                $price  = floatval(str_replace(',', '', $request->price[$i]));
                $qty    = floatval(str_replace(',', '', $request->qty[$i]));
                $amount = floatval(str_replace(',', '', $request->amount[$i]));

                // Create new sold items
                SoldItem::create([
                    'sale_id'    => $sale->id,
                    'product_id' => $productId,
                    'unit_id'    => $unitId,
                    'cont'       => 1,
                    'price'      => $price,
                    'qty'        => $qty,
                    'amount'     => $amount,
                ]);

                // Deduct stock for the new updated items
                recordStockMovement(
                    date('Y-m-d'),
                    auth()->user()->id,
                    $productId,
                    $request->location[$i],
                    "Sale Adjustment",
                    "Out",
                    $qty
                );

                $newItems[] = [
                    'product_id' => $productId,
                    'qty'        => $qty,
                    'price'      => $price,
                    'total'      => $amount,
                    'unit_id'    => $unitId,
                ];
            }

            if ($qrcodeSetting &&  $qrcodeSetting == 1) {
                // Sync with QR System
                $apiResponse = Http::post('https://thecaveqr.dukamax.online/api/cancel-order', [
                    'table_id' => $tableId,
                    'items'    => $newItems
                ]);

                if ($apiResponse->successful() && ($apiResponse['success'] ?? false)) {
                    DB::commit();
                    return redirect()->back()->with('success', 'Bill updated and QR system synced.');
                } else {
                    DB::rollBack();
                    return redirect()->back()->with('error', 'Failed to sync cancellation to QR system.');
                }
            }

            DB::commit();
            return redirect()->back()->with('success', 'Bill updated successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error("Cancel Order Error: " . $e->getMessage());

            return redirect()->back()->with('error', 'Something went wrong. Please try again.');
        }
    }


    public function billItemProcessor(
        $location,
        $table,
        $room,
        $waiter,
        $product_id,
        $qty,
        $amount
    ) {
        try {
            $sale = Sale::where("location_id", $location)
                ->where("table_id", $table)
                ->where("waiter_id", $waiter)
                ->where("status", "credit")
                ->first();

            $item = SoldItem::where("sale_id", $sale->id)
                ->where("product_id", $product_id)
                ->first();

            if ($item) {
                if ($item->qty > $qty) {
                    $item->qty -= $qty;
                    $item->amount -= $amount;
                    $item->update();
                } else {
                    $difference = $qty - $item->qty;
                    $item->delete();

                    if ($difference > 0) {
                        $this->billItemProcessor($location, $table, $room, $waiter, $product_id, $difference, $amount);
                    }
                }
            }
        } catch (Exception $e) {
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function store(Request $request)
    {
        DB::beginTransaction();
      //  try {
            $waiterData = Waiter::find($request->waiter);
            $profileData = BusinessProfile::where('selling_id', $waiterData->selling_id)->first();
            $tableData = Table::find($request->table);

            # Block the table
            if ($tableData && $tableData->flag == 1) {
                $tableData->update(['flag' => 2]);
            }

            $orderTotal = $this->cleanNumber($request->total); // Clean and convert total

            $orderArray = [
                "location" => [],
                "id" => [],
                "product" => [],
                "unit" => [],
                "price" => [],
                "qty" => [],
                "amount" => [],
            ];

            // Define incrementProductQuantity as a class method
            $this->incrementProductQuantity($orderArray, $request);

            # Get waiter table sale 
            $sale = Sale::where("waiter_id", $request->waiter)
                ->where("table_id", $request->table)
                ->where("status", "credit")
                ->where("due", ">", 0)
                ->first();

            if ($sale) { # If it exists increment the total, net, due and update status 
                $sale->total += $orderTotal;
                $sale->net += $orderTotal;
                $sale->due += $orderTotal;
                $sale->status = "credit";
                $sale->update();
            } else { # If it doesn't exist then create a new sale
                $sale = new Sale();
                $sale->date = date('Y-m-d');
                $sale->user_id = auth()->user()->id;
                $sale->waiter_id = $request->waiter;
                $sale->table_id = $request->table;
                $sale->total = $orderTotal;
                $sale->net = $orderTotal;
                $sale->due = $orderTotal;
                $sale->selling_id = $waiterData->selling_id;
                $sale->paid = 0;
                $sale->status = "credit";
                $sale->save();
            }

            # Get category data
            $categoryData = ProductCategory::where("name", "Menu")->first();

            if (!$categoryData) { # Redirect back if category is not found
                return back()->with('error', 'Category (Menu) does not exist.');
            }

            for ($i = 0; $i < count($orderArray["id"]); $i++) {
                # Get each product
                $productData = Product::where("id", $orderArray["id"][$i])
                    ->where("location_id", $orderArray["location"][$i])
                    ->first();

                # Get each unit
                $unitData = Unit::where("name", $orderArray["unit"][$i])->first();

                # Get each sold item
                $item = SoldItem::where("sale_id", $sale->id)
                    ->where("product_id", $orderArray["id"][$i])
                    ->first();

                if ($item) { # If it exists then increment the qty and amount
                    $item->qty += $orderArray["qty"][$i];
                    $item->amount += $orderArray["amount"][$i];
                    $item->update();
                } else { # If it doesn't exist then create it
                    $item = new SoldItem();
                    $item->sale_id = $sale->id;
                    $item->product_id = $orderArray["id"][$i];
                    $item->unit_id = $unitData->id;
                    $item->cont = 1;
                    $item->price = $orderArray["price"][$i];
                    $item->qty = $orderArray["qty"][$i];
                    $item->amount = $orderArray["amount"][$i];
                    $item->save();
                }

                if ($categoryData && MenuItems::where("product_id", $orderArray["id"][$i])->exists()) { # If menu has assigned products
                    # Get the menu assigned ingredients
                    $items = MenuItems::where("product_id", $orderArray["id"][$i])->get();

                    # Loop through each assigned ingredients
                    foreach ($items as $item) {
                        # Get each ingredient
                        $ingredient = Ingredient::find($item->ingredient_id);
                        if ($ingredient) { # If found decrement its qty and update
                            $ingredient->qty -= $orderArray["qty"][$i] * $item->qty;
                            $ingredient->update();
                        } else { # if not found go to the next item
                            continue;
                        }
                    }
                } else {
                    // Use the helper function to record stock movement and update stock
                    recordStockMovement(
                        date('Y-m-d'),                 // current date
                        auth()->user()->id,            // current user ID
                        $orderArray["id"][$i],         // product ID
                        $orderArray["location"][$i],   // location ID
                        "Sale",                        // reason
                        "Out",                         // status
                        $orderArray["qty"][$i]         // quantity
                    );
                }
            }

            // Get table by id
            $tableData = Table::find($request->table);

            if ($tableData) {
                $tableData->flag == 'occupied';
                $tableData->save();
            }

            DB::commit();

            $dateTime = date("Y-m-d H:i:s");
            $status = "end";
            $role = auth()->user()->role;

            // Initialize the order number variable
            $orderNumber = 0;

            // Get the current date and time in HH:MM:SS format
            $currentTime = date('H:i:s'); // This will return the current time, e.g., '17:27:41'
            $currentDate = date('Y-m-d'); // This will return the current date, e.g., '2024-09-20'

            // Check if the last generated date is stored in the session
            if (session()->has('last_generated_date')) {
                // Compare the current date with the last generated date
                if ($currentDate !== session('last_generated_date')) {
                    // If it's a new day, reset the order number to 0
                    $orderNumber = 0;

                    // Update the session with the current date
                    session(['last_generated_date' => $currentDate]);
                } else {
                    // If it's the same day, retrieve the existing order number from the session
                    $orderNumber = session('order_number', 0);
                }
            } else {
                // If there's no last generated date in the session, initialize it
                session(['last_generated_date' => $currentDate]);
            }

            // Increment the order number by 1
            $orderNumber += 1;

            // Store the updated order number back in the session for future reference
            session(['order_number' => $orderNumber]);

            return view("print-bill", compact(
                "role",
                "dateTime",
                "profileData",
                "orderArray",
                "orderTotal",
                "waiterData",
                "tableData",
                "orderNumber",
                "status"
            ));
        // } catch (Exception $e) {
        //     DB::rollBack(); // Rollback the transaction
        //     return response()->json(['error' => 'Something went wrong'], 500);
        // }
    }

    private function cleanNumber($value)
    {
        // Remove commas
        $value = str_replace(',', '', $value);

        // Check if the value has a decimal point and convert accordingly
        return is_numeric($value) ? (int) $value : 0;
    }


    private function incrementProductQuantity(&$orderArray, $request)
    {
        for ($i = 0; $i < count($request->id); $i++) {
            $location = $request->location[$i];
            $id = $request->id[$i];
            $product = $request->product[$i];
            $unit = $request->unit[$i];
            $qty = $this->cleanNumber($request->qty[$i]);
            $price = $this->cleanNumber($request->price[$i]);
            $amount = $this->cleanNumber($request->amount[$i]);

            $index = array_search([$location, $id], array_map(null, $orderArray["location"], $orderArray["id"]));
            if ($index !== false) {
                $orderArray["qty"][$index] += $qty;
                $orderArray["amount"][$index] += $amount;
            } else {
                $orderArray["location"][] = $location;
                $orderArray["id"][] = $id;
                $orderArray["product"][] = $product;
                $orderArray["unit"][] = $unit;
                $orderArray["price"][] = $price;
                $orderArray["qty"][] = $qty;
                $orderArray["amount"][] = $amount;
            }
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function vat($total)
    {
        try {
            $answer = (18 / 100) * $total;
            return $answer;
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        //
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        //
    }
}
