<?php

namespace App\Http\Controllers;

use App\Exports\RejectedStockExport;
use App\Exports\StocktackingExport;
use App\Imports\ProductImport;
use App\Imports\StockTackingImport;
use App\Models\BusinessProfile;
use App\Models\Damage;
use App\Models\DuplicateProduct;
use App\Models\License;
use App\Models\LoadedItems;
use App\Models\LoadVehicle;
use App\Models\Location;
use App\Models\Product;
use App\Models\ProductCategory;
use App\Models\Selling;
use App\Models\Stock;
use App\Models\StockMovement;
use App\Models\StockTacking;
use App\Models\StockTransfer;
use App\Models\Unit;
use App\Models\UnitAssigned;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Maatwebsite\Excel\Facades\Excel;
use App\Models\Storage; // 🔹 your Eloquent model

class ProductController extends Controller
{

    // public function __construct()
    // {
    //     $this->middleware('auth');
    // }
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        try {
            # Get location details
            $locationDetails = processLocationDetails();
            $sellingCount = $locationDetails['sellingCount'];
            $sellings = $locationDetails['sellings'];
            $suppliers = $locationDetails['suppliers'];
            $categories = ProductCategory::all();

            return view("products", compact("sellingCount", "sellings", "suppliers", "categories"));
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }
    public function stockResetView()
    {
        try {
            # Get location details
            $locationDetails = processLocationDetails();
            $sellingCount = $locationDetails['sellingCount'];
            $sellings = $locationDetails['sellings'];

            return view("stock-reset", compact("sellingCount", "sellings"));
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function stockReset(Request $request)
    {
        try {
            // Update the qty of all matching stocks in a single query
            Stock::where("location_id", $request->location)->update(['qty' => 0]);
            // Clear sessions.
            session()->forget(["sold_items", "sales_details"]);
            return back()->with("success", "Stock reset successfully!");
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function stockTackingView()
    {
        try {
            # Get location details
            $locationDetails = processLocationDetails();
            $sellingCount = $locationDetails['sellingCount'];
            $sellings = $locationDetails['sellings'];

            return view("stock-tacking", compact("sellingCount", "sellings"));
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function uploadProductExcel(Request $request)
    {
        // try {
        Excel::import(new ProductImport, $request->file);
        return back()->with("success", "Excel uploaded!!");
        // } catch (Exception $e) {
        //     // Log the error
        //     Log::error($e->getMessage());
        //     // Return a custom error response
        //     return response()->json(['error' => 'Something went wrong'], 500);
        // }
    }

    public function uploadStockTackingExcel(Request $request)
    {
        DB::beginTransaction(); // Start a transaction

        try {
            // Clear the stock tacking table
            StockTacking::truncate();

            // Upload the selected excel to stock tacking table
            Excel::import(new StockTackingImport, $request->file);

            // Get all stock tacking data in one query
            $stockTacking = StockTacking::all();

            // Loop through stock tacking data
            foreach ($stockTacking as $data) {
                $product = $data->product;
                $location_id = $request->location;

                // Get each product form products table.
                $productData = Product::where("product", $product)->first();

                // Get the stock and product information in a single query
                $stockData = Stock::where("product_id", $productData->id)
                    ->where("location_id", $location_id)
                    ->first();

                // Update the stock tacking table
                $data->oldqty = $stockData->qty;
                $data->buying = $productData->buying;
                $data->missing_value = $productData->buying * ($data->newqty - $data->oldqty);
                $data->update();

                $stockData->qty = $data->newqty;
                $stockData->update();
            }

            // Commit the transaction if everything is successful
            DB::commit();

            return back()->with("success", "Excel successfully uploaded & stock is updated!");
        } catch (Exception $e) {
            // Rollback the transaction in case of an error
            DB::rollback();
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function downloadStocktackingExcel(Request $request)
    {
        try {
            return Excel::download(new StocktackingExport($request), "StockTacking.xls");
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function downloadRejectedStockExcel(Request $request)
    {
        try {
            return Excel::download(new RejectedStockExport, "Rejected stock.xls");
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function deleteDuplicateProducts(Request $request)
    {
        try {
            $duplicates = DuplicateProduct::all();
            foreach ($duplicates as $duplicate) {
                $duplicate->delete();
            }
            return back()->with("success", "Rejected products deleted successfully!!");
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function generateBarcode()
    {
        $code = request()->barcode;
        return \Milon\Barcode\Facades\DNS1DFacade::getBarcodePNG($code, 'C39', 1, 20);
    }

    public function barcodeView()
    {
        //dd(\Milon\Barcode\Facades\DNS1DFacade::getBarcodePNG('+barcode+', 'C39'));
        try {
            # Get location details
            $locationDetails = processLocationDetails();
            $sellingCount = $locationDetails['sellingCount'];
            $sellings = $locationDetails['sellings'];

            return view("barcode", compact('sellingCount', 'sellings'));
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function deleteProductView()
    {
        try {
            # Get location details
            $locationDetails = processLocationDetails();
            $sellingCount = $locationDetails['sellingCount'];
            $sellings = $locationDetails['sellings'];

            return view("delete-product", compact("sellingCount", "sellings"));
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function updateProductView()
    {
        //try {
        # Get location details
        $locationDetails = processLocationDetails();
        $sellingCount = $locationDetails['sellingCount'];
        $sellings = $locationDetails['sellings'];
        $batchStatus = BusinessProfile::first()->value('batch_no');

        $categories = ProductCategory::all();
        $units = Unit::all();

        return view("update-product-details", compact(
            "sellingCount",
            "sellings",
            "categories",
            "units",
            "batchStatus"
        ));
        // } catch (Exception $e) {
        //     // Log the error
        //     Log::error($e->getMessage());
        //     // Return a custom error response
        //     return response()->json(['error' => 'Something went wrong'], 500);
        // }
    }

    public function updateProductPriceView()
    {
        try {
            # Get location details
            $locationDetails = processLocationDetails();
            $sellingCount = $locationDetails['sellingCount'];
            $sellings = $locationDetails['sellings'];

            $units = Unit::all();

            return view("update-price", compact(
                "sellings",
                "sellingCount",
                "units"
            ));
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function stockListView(Request $request)
    {
        // try {
        # Get location details
        $locationDetails = processLocationDetails();
        $sellingCount = $locationDetails['sellingCount'];
        $sellings = $locationDetails['sellings'];
        $locationId = null;
        $products = collect(); // Creates an empty collection

        if ($sellingCount == 1) {
            $locationId = $sellings[0]->location_id;
        } else {
            if ($request->has('location')) {
                $locationId = $request->location;
            }
        }

        if ($locationId != null) {
            # Initiate variable
            $categoryIds = [2, 3, 5];

            $categories = ProductCategory::whereIn('id', $categoryIds)
                ->orWhereIn('parent_id', $categoryIds)
                ->get();

            $products = Product::where("location_id", $locationId)->whereIn('category_id', $categories->pluck('id')->toArray())->with(["supplier", "category"])->orderBy('product', 'asc')->get();
        }

        return view("stock-list", compact(
            "products",
            "locationId",
            "sellingCount",
            "sellings"
        ));
        // } catch (Exception $e) {
        //     // Log the error
        //     Log::error($e->getMessage());
        //     // Return a custom error response
        //     return response()->json(['error' => 'Something went wrong'], 500);
        // }
    }

    public function priceListView(Request $request)
    {
        try {
            // Initialize variables.
            $location_id = null; // Initialize location_id as null.
            $products = collect(); // Initialize products as an empty collection.

            # Get location details
            $locationDetails = processLocationDetails();
            $sellingCount = $locationDetails['sellingCount'];
            $sellings = $locationDetails['sellings'];

            if ($sellingCount == 1) {
                // If there is only one selling location, fetch products for that location.
                $products = Product::where("location_id", $sellings[0]->location_id)
                    ->with("stock.location")
                    ->orderBy('product', 'asc')
                    ->get();
            } else {
                // If there are multiple selling locations and a location is specified in the request.
                if ($request->has('location')) {
                    $location_id = $request->location;
                    $products = Product::where("location_id", $request->location)
                        ->with("stock.location")
                        ->orderBy('product', 'asc')
                        ->get();
                }
            }

            $licenseData = License::select('tra')->first()->value('tra');
            $tra = null;

            if ($licenseData != null && auth()->user()->name == 'Admin') {
                $tra = $licenseData / 100;
            }

            // Return the price-list view with the necessary data.
            return view("price-list", compact("products", "location_id", "sellings", "sellingCount", "tra"));
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response if an exception occurs.
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }


    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */

    public function stockTransferView()
    {
        try {
            # Get location details
            $locationDetails = processLocationDetails();
            $sellingCount = $locationDetails['sellingCount'];
            $sellings = $locationDetails['sellings'];

            return view("stock-transfer", compact("sellings", "sellingCount"));
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function damageStockView()
    {
        try {
            # Get location details
            $locationDetails = processLocationDetails();
            $sellingCount = $locationDetails['sellingCount'];
            $sellings = $locationDetails['sellings'];

            return view("damage-stock", compact("sellings", "sellingCount"));
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        DB::beginTransaction();

        try {
            // Abort if the product already exists at this location
            if (Product::where('product', ucwords($request->product))
                ->where('location_id', $request->location)
                ->exists()
            ) {
                return back()->with('error', "{$request->product} already exists in {$request->location}");
            }

            $category = ProductCategory::find($request->category);

            // Validate common fields
            $request->validate([
                'product' => 'required',
                'category' => 'required',
                'location' => 'required',
            ]);

            $product = new Product();
            $product->barcode = $request->barcode ?? str_pad(rand(0, 9999999999999), 13, '0', STR_PAD_LEFT);
            $product->product = ucwords($request->product);
            $product->category_id = $request->category;
            $product->buying = $request->buying ?? 0;
            $product->supplier_id = $request->supplier ?? null;
            $product->location_id = $request->location;

            // Handle image for normal products
            if ($request->hasFile('image')) {
                $image = $request->file('image');
                $fileName = uniqid('image_') . '.' . $image->getClientOriginalExtension();
                $image->storeAs('public/images', $fileName);
                $product->image = $fileName;
            }

            $product->save();

            // Handle units
            for ($i = 0; $i < count($request->unit); ++$i) {
                $unit = new UnitAssigned();
                $unit->product_id = $product->id;
                $unit->unit_id = $request->unit[$i];
                $unit->unit_cont = $request->content[$i];
                $unit->selling = $request->selling[$i];
                $unit->location_id = $request->location;
                $unit->save();
            }

            // Handle stock and batches
            $batchNo = $request->batch ?? null;
            $expDate = $request->exp_date ?? null;
            $qty = $request->qty ?? 0;

            // Record stock movement
            recordStockMovement(
                date('Y-m-d'),
                Auth::id(),
                $product->id,
                $request->location,
                "Registered",
                "In",
                $qty,
                $batchNo,
                $expDate
            );

            DB::commit();

            return back()->with("success", "Product successfully Registered!!");
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error($e->getMessage());
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function updateProduct(Request $request)
    {
        $batchStatus = BusinessProfile::first()->value('batch_no');

        $location_id = $request->location;

        DB::beginTransaction();
        try {
            foreach ($request->id as $index => $productId) {
                $product = Product::findOrFail($productId);

                // 🔹 Handle image
                $fileName = $product->image;
                if ($request->hasFile("image.$index")) {
                    $image = $request->file("image")[$index];
                    if ($fileName && Storage::exists("public/images/$fileName")) {
                        Storage::delete("public/images/$fileName");
                    }
                    $fileName = uniqid('image_') . '.' . $image->getClientOriginalExtension();
                    $image->storeAs("public/images", $fileName);
                }

                // 🔹 Update product details
                $product->update([
                    'image'       => $fileName,
                    'barcode'     => $request->barcode[$index],
                    'product'     => ucwords($request->product[$index]),
                    'buying'      => unformatNumber($request->buying[$index] ?? 0),
                    'category_id' => $request->category[$index] !== "null" ? $request->category[$index] : $product->category_id,
                ]);

                if ($batchStatus == 0) {
                    $stock = Stock::where('product_id', $productId)
                        ->where('location_id', $location_id)->first();

                    $oldQty = $stock?->qty ?? 0;
                    $diff = $request->total_stock[$index] - $oldQty;

                    if ($diff !== 0 || !$stock) {
                        // Record movement and update/create stock
                        recordStockMovement(
                            now()->toDateString(),
                            auth()->id(),
                            $productId,
                            $location_id,
                            "Update Product",
                            $diff >= 0 ? "In" : "Out",
                            abs($diff),
                        );
                    }
                } else {
                    // 🔹 Handle Batches (Stocks)
                    $submittedBatches = $request->batch_no[$productId] ?? [];

                    // Delete removed batches
                    Stock::where('product_id', $productId)
                        ->where('location_id', $location_id)
                        ->whereNotIn('batch_no', $submittedBatches)
                        ->delete();

                    // Update/create submitted batches
                    foreach ($submittedBatches as $bIndex => $batchNo) {
                        $qty     = (int) ($request->qty[$productId][$bIndex] ?? 0);
                        $expDate = $request->exp[$productId][$bIndex] ?? null;

                        $stock = Stock::where('product_id', $productId)
                            ->where('location_id', $location_id)
                            ->where('batch_no', $batchNo)
                            ->first();

                        $oldQty = $stock?->qty ?? 0;
                        $diff = $qty - $oldQty;

                        if ($diff !== 0 || !$stock) {
                            // Record movement and update/create stock
                            recordStockMovement(
                                now()->toDateString(),
                                auth()->id(),
                                $productId,
                                $location_id,
                                "Update Product",
                                $diff >= 0 ? "In" : "Out",
                                abs($diff),
                                $batchNo,
                                $expDate
                            );
                        }
                    }
                }

                // 🔹 Handle Units
                $submittedUnits = $request->unit_id[$productId] ?? [];

                // Delete removed units
                UnitAssigned::where('product_id', $productId)
                    ->where('location_id', $location_id)
                    ->whereNotIn('unit_id', $submittedUnits)
                    ->delete();

                // Update/create submitted units
                foreach ($submittedUnits as $uIndex => $unitId) {
                    UnitAssigned::updateOrCreate(
                        [
                            'product_id'  => $productId,
                            'unit_id'     => $unitId,
                            'location_id' => $location_id,
                        ],
                        [
                            'unit_cont' => $request->unit_cont[$productId][$uIndex] ?? 1,
                            'selling'   => unformatNumber($request->selling[$productId][$uIndex] ?? 0),
                        ]
                    );
                }
            }

            DB::commit();
            return back()->with('success', 'Products updated successfully!');
        } catch (\Throwable $e) {
            DB::rollBack();
            return back()->with('error', 'Failed to update: ' . $e->getMessage());
        }
    }

    public function updatePrice(Request $request)
    {
        $data = $request->all();
        try {

            DB::beginTransaction(); // Start a database transaction

            foreach ($request->product as $i => $productName) {
                // Find a product by name and location in
                $item = Product::where("product", $productName)
                    ->where("location_id", $request->location)
                    ->first();

                // Check if the product exists
                if ($item) {
                    // Find unit assignments for the product and location
                    $unitAssigned = UnitAssigned::where("product_id", $item->id)
                        ->where("location_id", $request->location);

                    // If there's only one unit assignment, update it
                    if ($unitAssigned->count() == 1) {
                        $unitAssigned->update([
                            'unit_id' => $data['unit_' . $item->id][0], //$request->unit[$i],
                            'unit_cont' => $data['cont_' . $item->id][0], //$request->cont[$i],
                            'selling' => $data['sell_' . $item->id][0], //$request->sell[$i],
                        ]);
                    } else {
                        // If there are multiple unit assignments, update them individually
                        $units = $unitAssigned->get();
                        //dd($units);
                        foreach ($units as $u => $unit) {
                            if ($data['unit_' . $item->id][$u] !== null) {
                                $unit->update([
                                    'unit_id' => $data['unit_' . $item->id][$u], //$request->unit[$u],
                                    'unit_cont' => $data['cont_' . $item->id][$u], //$request->cont[$u],
                                    'selling' => $data['sell_' . $item->id][$u] //$request->sell[$u],
                                ]);
                            }
                        }
                    }

                    // Update the product's buying price
                    $item->update([
                        'buying' => $request->buying[$i],
                    ]);
                }
            }

            DB::commit(); // Commit the database transaction

            return back()->with('success', 'Successfully updated!!');
        } catch (\Exception $e) {
            DB::rollback(); // Rollback the database transaction on error
            Log::error($e->getMessage());
            return response()->json(['error' => $e->getMessage()], 500);
        }
    }

    public function storeDamageStock(Request $request)
    {
        // try {
        DB::beginTransaction(); // Start transaction

        $date = $request->date;
        $user = Auth()->id();

        $damageData = [];

        for ($i = 0; $i < count($request->id); $i++) {
            $product_id = $request->id[$i];
            $qty = $request->qty[$i];
            $location_id = $request->location[$i];
            $batch_no = $request->batch_no[$i] !== "Null" ? $request->batch_no[$i] : null;

            $damageData[] = [
                'date' => $date,
                'user_id' => $user,
                'product_id' => $product_id,
                'reason' => $request->reason[$i],
                'unit_id' => $request->unit[$i],
                'price' => $request->price[$i],
                'cont' => $request->cont[$i],
                'location_id' => $location_id,
                'qty' => $qty,
                'batch_no' => $batch_no,
                'created_at' => now(),
                'updated_at' => now(),
            ];

            // Record stock movement with batch
            recordStockMovement(
                $date,
                $user,
                $product_id,
                $location_id,
                "Damage",
                "Out",
                $qty,
                $batch_no
            );
        }

        Damage::insert($damageData);

        DB::commit();

        return back()->with("success", "Successfully posted!!");
        // } catch (Exception $e) {
        //     DB::rollback();
        //     Log::error($e->getMessage());
        //     return response()->json(['error' => 'Something went wrong'], 500);
        // }
    }

    public function storeStockTransfer(Request $request)
    {
        //try {
        DB::beginTransaction();

        $date  = $request->date;
        $userId = auth()->user()->id;
        $productionLocationIds = Location::where("type", "production")->pluck("id")->toArray();

        $flag = null;

        // loop through all products
        for ($i = 0; $i < count($request->product_id); ++$i) {
            $productId    = $request->product_id[$i];
            $qty          = $request->qty[$i] * $request->cont[$i];
            $fromLocation = $request->from[$i];
            $toLocation   = $request->to[$i];
            $unitId       = $request->unit_id[$i];
            $cont         = $request->cont[$i];
            $price        = $request->price[$i];
            $batchNo      = $request->batch_no[$i] ?? null;

            // 1️⃣ Record the stock transfer itself
            $transfer = StockTransfer::create([
                'date'        => $date,
                'location_id' => $request->area,
                'user_id'     => $userId,
                'product_id'  => $productId,
                'batch_no'    => $batchNo,
                'unit_id'     => $unitId,
                'cont'        => $cont,
                'from'        => $fromLocation,
                'to'          => $toLocation,
                'qty'         => $qty,
                'price'       => $price,
            ]);

            if (!$flag) {
                $flag = $transfer->created_at;
            }

            // 2️⃣ Handle "from" location (Out)
            if (!in_array($fromLocation, $productionLocationIds)) {
                $fromStock = Stock::where('product_id', $productId)
                    ->where('location_id', $fromLocation)
                    ->when($batchNo, fn($q) => $q->where('batch_no', $batchNo))
                    ->first();

                if ($fromStock) {
                    recordStockMovement($date, $userId, $productId, $fromLocation, 'Stock Transfer', 'Out', $qty, $batchNo);
                }
            }

            // 3️⃣ Handle "to" location (In)
            $existingToProduct = Product::where('product', Product::find($productId)->product)
                ->where('location_id', $toLocation)
                ->first();

            if ($existingToProduct) {
                $toStock = Stock::where('product_id', $existingToProduct->id)
                    ->where('location_id', $toLocation)
                    ->when($batchNo, fn($q) => $q->where('batch_no', $batchNo))
                    ->first();

                if ($toStock && !in_array($toLocation, $productionLocationIds)) {
                    recordStockMovement($date, $userId, $existingToProduct->id, $toLocation, 'Stock Transfer', 'In', $qty, $batchNo);
                }
            } else {
                // 4️⃣ Create new product at "to" location
                $oldProduct = Product::find($productId);

                $newProduct = Product::create([
                    'barcode'     => $oldProduct->barcode,
                    'image'       => $oldProduct->image,
                    'product'     => $oldProduct->product,
                    'category_id' => $oldProduct->category_id,
                    'buying'      => $oldProduct->buying,
                    'supplier_id' => $oldProduct->supplier_id,
                    'location_id' => $toLocation
                ]);

                // Copy units
                $oldUnits = UnitAssigned::where('product_id', $productId)
                    ->where('location_id', $fromLocation)
                    ->get();

                foreach ($oldUnits as $oldUnit) {
                    UnitAssigned::create([
                        'product_id'  => $newProduct->id,
                        'unit_id'     => $oldUnit->unit_id,
                        'unit_cont'   => $oldUnit->unit_cont,
                        'selling'     => $oldUnit->selling,
                        'location_id' => $toLocation
                    ]);
                }

                // Copy stock
                $oldStock = Stock::where('product_id', $productId)
                    ->where('location_id', $fromLocation)
                    ->first();

                $newStock = Stock::create([
                    'product_id' => $newProduct->id,
                    'location_id' => $toLocation,
                    'unit_id'    => $oldStock->unit_id,
                    'min_cont'   => $oldStock->min_cont,
                    'min_qty'    => $oldStock->min_qty,
                    'qty'        => 0,
                    'exp'        => $oldStock->exp,
                    'exp_not'    => $oldStock->exp_not,
                    'batch_no'   => $batchNo, // ✅ batch assigned here too
                ]);

                recordStockMovement($date, $userId, $newProduct->id, $toLocation, 'Stock Transfer', 'In', $qty, $batchNo);
            }
        }

        // 5️⃣ Prepare slip
        $selling     = Selling::where('location_id', $request->area)->first();
        $profileData = BusinessProfile::where('selling_id', $selling->id)->first();
        $transferedItems = StockTransfer::where('created_at', $flag)
            ->with('location', 'product', 'unit')
            ->get();

        DB::commit();

        return view('transfer-slip', compact('transferedItems', 'profileData', 'flag'));
        // } catch (\Exception $e) {
        //     DB::rollBack();
        //     Log::error($e->getMessage());
        //     return response()->json(['error' => 'Something went wrong', 'message' => $e->getMessage()], 500);
        // }
    }


    public function deleteStockTransfer(Request $request)
    {
        DB::beginTransaction();

        //try {
        $ids = $request->ids;

        if (empty($ids)) {
            return response()->json(['deleted' => [], 'message' => 'No IDs provided'], 400);
        }

        $deletedIds = [];
        $productionLocationIds = Location::where("type", "production")->pluck("id")->toArray();

        $stockTransfers = StockTransfer::whereIn('id', $ids)->get();

        foreach ($stockTransfers as $transfer) {
            $productId = $transfer->product_id;
            $fromLocation = (int)$transfer->from;
            $toLocation = (int)$transfer->to;
            $unitId = $transfer->unit_id;
            $cont = $transfer->cont;

            // If this transfer has multiple batches stored in a related table (or comma-separated)
            $batches = json_decode($transfer->batches, true); // Example: [{"batch_no":"#900","qty":50,"exp":null},...]

            if (!empty($batches)) {
                foreach ($batches as $batch) {
                    $qty = (int)$batch['qty'] * $cont;
                    $batchNo = $batch['batch_no'] ?? null;
                    $expDate = $batch['exp'] ?? null;

                    // Reverse 'from' location: add stock back
                    recordStockMovement(
                        $transfer->date,
                        Auth::id(),
                        $productId,
                        $fromLocation,
                        'Stock Transfer Deleted',
                        'In',
                        $qty,
                        $batchNo,
                        $expDate
                    );

                    // Reverse 'to' location: remove stock
                    recordStockMovement(
                        $transfer->date,
                        Auth::id(),
                        $productId,
                        $toLocation,
                        'Stock Transfer Deleted',
                        'Out',
                        $qty,
                        $batchNo,
                        $expDate
                    );
                }
            } else {
                // Single batch or no batch info
                $qty = (int)$transfer->qty * $cont;
                $batchNo = $transfer->batch_no;
                $expDate = $transfer->exp;

                recordStockMovement(
                    $transfer->date,
                    Auth::id(),
                    $productId,
                    $fromLocation,
                    'Stock Transfer Deleted',
                    'In',
                    $qty,
                    $batchNo,
                    $expDate
                );

                recordStockMovement(
                    $transfer->date,
                    Auth::id(),
                    $productId,
                    $toLocation,
                    'Stock Transfer Deleted',
                    'Out',
                    $qty,
                    $batchNo,
                    $expDate
                );
            }

            // Check if product at 'to' location was created by this transfer
            $originalProduct = Product::find($productId);
            $toProduct = Product::where('product', $originalProduct->product)
                ->where('location_id', $toLocation)
                ->first();

            if ($toProduct && $toProduct->id !== $productId) {
                UnitAssigned::where('product_id', $toProduct->id)
                    ->where('location_id', $toLocation)
                    ->delete();

                Stock::where('product_id', $toProduct->id)
                    ->where('location_id', $toLocation)
                    ->delete();

                $toProduct->delete();
            }

            // Delete the transfer record itself
            $transfer->delete();
            $deletedIds[] = $transfer->id;
        }

        DB::commit();

        return response()->json([
            'deleted' => $deletedIds,
            'message' => 'Stock transfer(s) fully reversed successfully.'
        ]);
        // } catch (\Exception $e) {
        //     DB::rollBack();
        //     Log::error('Error reversing stock transfer: ' . $e->getMessage());
        //     return response()->json(['deleted' => [], 'message' => 'Something went wrong.'], 500);
        // }
    }

    public function deleteDamageStock(Request $request)
    {
        DB::beginTransaction();

        try {
            $ids = $request->ids;

            if (empty($ids)) {
                return response()->json(['deleted' => [], 'message' => 'No IDs provided'], 400);
            }

            $deletedIds = [];

            // Retrieve damage records
            $damages = Damage::whereIn('id', $ids)->get();

            foreach ($damages as $damage) {
                // Reverse stock using batch info if available
                recordStockMovement(
                    now()->toDateString(),          // date
                    Auth::id(),                     // userId
                    $damage->product_id,            // productId
                    $damage->location_id,           // locationId
                    'Damage stock deleted',         // reason
                    'In',                           // status (add back to stock)
                    $damage->qty,                   // qty
                    $damage->batch_no,              // batchNo
                    $damage->exp_date ?? null       // expDate if exists
                );

                // Delete the damage record
                $damage->delete();
                $deletedIds[] = $damage->id;
            }

            DB::commit();

            return response()->json([
                'deleted' => $deletedIds,
                'message' => 'Damage stock reversed successfully'
            ]);
        } catch (Exception $e) {
            DB::rollBack();
            Log::error('Error reversing damaged stock: ' . $e->getMessage());
            return response()->json(['deleted' => [], 'message' => 'Something went wrong.'], 500);
        }
    }

    public function searchProduct(Request $request)
    {
        //  try {
        $locationId = $request->get('locationId');
        if (!$locationId) {
            return response()->json([], 200);
        }

        // Extract values safely
        $barcode = $request->get('barcode');
        $query   = $request->get('query');

        // Step 1: Get unique products in that location
        $products = Product::where('location_id', $locationId)
            ->when($barcode, fn($q) => $q->where('barcode', $barcode))
            ->when($query, fn($q) => $q->where('product', 'LIKE', "%{$query}%"))
            ->take(20)
            ->get();

        // Step 2: Eager load relations
        $products->load([
            'unit.unit',
            'category',
            'stock' => fn($q) => $q->where('location_id', $locationId)->orderBy('exp_date', 'asc')
        ]);

        // Step 3: Transform for frontend
        $transformed = $products->map(function ($product) {
            $batches = $product->stock->filter(fn($s) => $s->batch_no)->values();

            return [
                'id'        => $product->id,
                'location_id' => $product->location_id,
                'product'   => $product->product,
                'barcode'   => $product->barcode,
                'description' => $product->description,
                'image'     => $product->image,
                'category'  => $product->category?->name ?? null,
                'buying'    => $product->buying,
                'totalQty'  => $product->stock->sum('qty'),
                'unit'      => $product->unit->map(fn($u) => [
                    'id'        => $u->unit_id,
                    'name'      => $u->unit->name ?? '',
                    'unit_cont' => $u->unit_cont ?? 1,
                    'selling'   => $u->selling
                ]),
                'batches'   => $batches->map(fn($s) => [
                    'batch_no' => $s->batch_no,
                    'qty'      => $s->qty,
                    'exp'      => $s->exp_date
                ])
            ];
        });

        return response()->json($transformed);
        // } catch (\Exception $e) {
        //     Log::error($e->getMessage());
        //     return response()->json(['error' => 'Something went wrong'], 500);
        // }
    }

    public function productBarcode(Request $request)
    {
        try {
            // Get the barcode and location from the request
            $barcode = $request->get('barcode');
            // $location_id = $request->get('locationId');

            // Retrieve the products with their associated units and stocks
            $results = Product::where('barcode', $barcode)  // Apply the barcode filter
                // ->where('location_id', $location_id)  // Apply the location filter
                ->with('unit.unit', 'stock.unit', 'category')
                ->get();

            // Return the results as a JSON response
            return $results;
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function VehicleProduct(Request $request)
    {
        try {
            // Get the query and vehicle from the request
            $query = $request->get('query');
            $vehicle = $request->get('vehicle');

            // Retrieve the ID of the loaded vehicle matching the provided vehicle and "Loaded" status
            $loadVehicleId = LoadVehicle::where('vehicle', $vehicle)
                ->where('status', 'Loaded')
                ->value('id');

            // Retrieve the loaded items associated with the load vehicle, matching the query
            $items = LoadedItems::with('units')
                ->where('load_vehicle_id', $loadVehicleId)
                ->where('product', 'LIKE', "%$query%")
                ->get();

            // Return the items as a JSON response
            return response()->json($items);
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function loadedProduct(Request $request)
    {
        try {
            // Get the query and location from the request
            $query = $request->get('query');
            $location = $request->get('location');

            // Join the LoadVehicle and LoadedItems models using the defined relationship
            $results = LoadVehicle::join('loaded_items', 'load_vehicles.load_vehicle_id', '=', 'loaded_items.id')
                ->where('loaded_items.product', 'like', "%$query%") // Apply the query filter
                ->where('loaded_items.location', $location) // Apply the location filter
                ->get();

            // Return the results as a JSON response
            return response()->json($results);
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function unitDetails(Request $request)
    {
        try {
            // Get the product, location, and unit from the request
            $product_id = $request->get('productId');
            $unit_id = $request->get('unitId');
            $location_id = $request->get('locationId');

            $results = UnitAssigned::with("product", "unit")
                ->where('product_id', $product_id)  // Apply the product filter
                ->where('unit_id', $unit_id);  // Apply the unit filter

            $results = $results->when($location_id, function ($query) use ($location_id) {
                return $query->where('location_id', $location_id);  // Apply the location filter if $location_id is specified
            });

            $results = $results->first();

            // Return the results as a JSON response
            return response()->json($results);
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function getProduct(Request $request)
    {
        // Get the query and location from the request
        $query = $request->get('query');
        $locationData = Location::where('id', $request->location)->first();
        $businessName = $locationData->name;

        // Retrieve products with associated units, filtered by query and location
        $results = Product::with('unit', 'stock.unit')
            ->where('product', 'like', "%$query%") // Apply the query filter
            ->take(15) // Limit the number of results to 15
            ->get();

        // Return the results and business name as a JSON response
        return response()->json([$results, $businessName]);
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        //
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */

    public function deleteProduct(Request $request)
    {
        DB::beginTransaction(); // Start transaction

        try {
            $productIds = $request->product_id ?? []; // array of product IDs
            $locationIds = $request->location_id ?? []; // array of location IDs
            $batchNos = $request->batch_no ?? []; // array of batch numbers (can be empty)

            foreach ($productIds as $index => $productId) {
                $locId = $locationIds[$index] ?? null;   // scalar location ID
                $batchNo = $batchNos[$index] ?? null;   // scalar batch number

                // Get the stock record for this product and batch (if any)
                $stock = Stock::where('product_id', $productId)
                    ->where('location_id', $locId)
                    ->when($batchNo, fn($q) => $q->where('batch_no', $batchNo))
                    ->first();

                if ($stock) {
                    // Record stock movement for "Out"
                    recordStockMovement(
                        date('Y-m-d'),
                        auth()->id(),
                        $productId,
                        $locId,
                        'Product Deleted',
                        'Out',
                        $stock->qty,
                        $stock->batch_no,
                        $stock->exp_date
                    );

                    // Delete this stock entry
                    $stock->delete();
                }

                // Check if any stock remains for this product in this location
                $remainingStock = Stock::where('product_id', $productId)
                    ->where('location_id', $locId)
                    ->count();

                if ($remainingStock === 0) {
                    // If no stock left, delete the product and its unit assignments
                    Product::where('id', $productId)->delete();
                    UnitAssigned::where('product_id', $productId)->delete();
                }
            }

            DB::commit(); // Commit transaction
            return back()->with('success', 'Successfully deleted selected product(s)');
        } catch (\Exception $e) {
            DB::rollBack(); // Rollback on error
            Log::error('Delete Product Error: ' . $e->getMessage());
            return response()->json(['success' => 'Something went wrong'], 500);
        }
    }


    public function unitsUpdates(Request $request, $id)
    {
        // Validation (ensure all arrays are provided)
        $request->validate([
            'unit' => 'required|array',
            'cont' => 'required|array',
            'buy' => 'required|array',
            'price' => 'required|array',
        ]);

        try {
            DB::beginTransaction();

            // Get the array of product IDs and the location from the request
            $units = $request->unit;
            $conts = $request->cont;
            $buying = $request->buy;
            $prices = $request->price;

            // Fetch product once
            $productData = Product::find($id);
            if (!$productData) {
                return back()->with("error", "Product not found");
            }

            $location = $productData->location_id;

            // Get the current list of units for the product
            $currentUnits = UnitAssigned::where("product_id", $id)->pluck('unit_id')->toArray();

            // Iterate over the incoming data
            for ($i = 0; $i < count($units); $i++) {
                $unitAssigned = UnitAssigned::where("product_id", $id)
                    ->where("unit_id", $units[$i])
                    ->first();

                if ($unitAssigned) {
                    // Update existing unit assignment
                    $unitAssigned->update([
                        'unit_cont' => $conts[$i],
                        'selling' => $prices[$i],
                    ]);
                } else {
                    // Create new unit assignment
                    UnitAssigned::create([
                        'product_id' => $id,
                        'unit_id' => $units[$i],
                        'unit_cont' => $conts[$i],
                        'selling' => $prices[$i],
                        'location_id' => $location,
                    ]);
                }

                // Remove from the list of current units (since it's updated or created)
                if (($key = array_search($units[$i], $currentUnits)) !== false) {
                    unset($currentUnits[$key]);
                }

                // Ensure no division by zero for unit content
                if ($conts[$i] > 0) {
                    // Calculate the new buying price
                    $productData->buying = $buying[$i] / $conts[$i];
                    $productData->save();
                }
            }

            // Delete the remaining units in currentUnits (those not in incoming data)
            if (!empty($currentUnits)) {
                UnitAssigned::where('product_id', $id)
                    ->whereIn('unit_id', $currentUnits)
                    ->delete();
            }

            DB::commit();

            return redirect()->back()->with('success', $productData->product . ' Units updated successfully');
        } catch (\Throwable $e) {
            // If something goes wrong rollback the transaction
            DB::rollBack();

            // Log the error
            Log::error($e->getMessage());

            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function customPricelist()
    {
        try {
            $units = UnitAssigned::where('unit_cont', 1)->with('product')->get();
            return view('custom-pricelist', compact('units'));
        } catch (\Exception $e) {
            return back()->with('error', 'Something went wrong.');
        }
    }

    public function updateBarcode(Request $request)
    {
        $request->validate([
            'barcode' => 'required|string|size:13',
            'product_id' => 'required|integer|exists:products,id',
        ]);

        $product = Product::find($request->product_id);

        if (!$product) {
            return response()->json(['success' => false, 'message' => 'Product not found.'], 404);
        }

        $product->barcode = $request->barcode;
        $product->save();

        return response()->json(['success' => true, 'message' => 'Barcode updated successfully!']);
    }

    public function getByCategory($id)
    {
        $products = Product::where('category_id', $id)->get();

        // Return JSON for AJAX
        return response()->json($products);
    }

    public function checkBatchStatus(Request $request)
    {
        $locationId = $request->location_id; // incoming value from AJAX

        // Try to find it in the `sellings` table first
        $selling = Selling::where('location_id', $locationId)->first();

        if ($selling) {
            $sellingId = $selling->id;
        } else {
            // Otherwise, try the `storages` table
            $storage = Storage::where('location_id', $locationId)->first();
            $sellingId = $storage->selling_id;
        }

        $batchNo = $sellingId ? getBatchNo($sellingId) : 0;

        return response()->json([
            'batch_no' => $batchNo
        ]);
    }
}
