<?php

namespace App\Http\Controllers;

use App\Models\Account;
use App\Models\BusinessProfile;
use App\Models\Customer;
use App\Models\Debts;
use App\Models\Product;
use App\Models\ProductionOrder;
use App\Models\ProductionOrderItem;
use App\Models\Selling;
use App\Models\Stock;
use App\Models\StockMovement;
use App\Models\Transaction;
use App\Models\UnitAssigned;
use Carbon\Carbon;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class ProductionController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        try {
            // Get location details
            $locationDetails = processLocationDetails();
            $sellingCount = $locationDetails['sellingCount'];
            $sellings = $locationDetails['sellings'];
            $accounts = $locationDetails['accounts'];

            return view('production-order', compact(
                'sellingCount',
                'sellings',
                'accounts'
            ));
        } catch (\Exception $e) {
            // Redirect or return an error response
            return back()->with('error', 'Failed to load production order details.');
        }
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function cancelProductionOrder()
    {
        return redirect()->route('production.order');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        //start transaction
        DB::beginTransaction();

        try {
            $inputs = $request->all();

            // Count the number of rows (assuming all arrays have the same length)
            $rowCount = count($inputs['id']);

            $items = [];

            for ($i = 0; $i < $rowCount; $i++) {
                $items[] = [
                    'id'          => $inputs['id'][$i],
                    'product'     => $inputs['product'][$i],
                    'unit'        => $inputs['unit'][$i],
                    'cont'        => $inputs['cont'][$i],
                    'size'        => $inputs['size'][$i],
                    'rangi'       => $inputs['rangi'][$i],
                    'main'        => $inputs['main'][$i],
                    'sub'         => $inputs['sub'][$i],
                    'qty'         => $inputs['qty'][$i],
                    'price'       => $inputs['price'][$i],
                    'amount'      => $inputs['amount'][$i],
                ];
            }

            $date         = $request->date;
            $total        = $request->total;
            $type         = $request->type;
            $genre        = $request->genre;
            $customerName = $request->customer;
            $paid         = $request->paid;
            $due          = $request->due;
            $expectation  = $request->expectation;
            $locationId   = $request->location;
            $accountId    = $request->account;

            # Get the selling id 
            $sellingId = Selling::where('location_id', $locationId)->first()->value('id');

            // Initialize variables
            $customerId = null;

            if (!empty($customerName)) {
                $customer = Customer::firstOrCreate(
                    ['name' => $customerName],
                    [
                        'address'     => 'N/A',
                        'phone'       => 'N/A',
                        'limit'       => 0,
                        'selling_id' => $sellingId,
                    ]
                );

                $customerId = $customer->id;
            }

            // Compute a week from today
            $weekFromToday = Carbon::parse($request->date)->addWeek()->toDateString();

            // Compute 2 weeek from today
            $twoWeeksFromToday = Carbon::parse($request->date)->addWeeks(2)->toDateString();

            // Compute 3 weeek from today
            $threeWeeksFromToday = Carbon::parse($request->date)->addWeeks(3)->toDateString();

            // Get the current date
            $currentDate = Carbon::now();

            // Add one month to the current date
            $oneMonthLater = $currentDate->addMonth();

            // Format the result as a string
            $oneMonthLaterFormatted = $oneMonthLater->toDateString();

            // Initiate varibale
            $expectation = null;

            // check if expectation is 1,2,3,4
            if ($request->expectation == 1) {
                $expectation = $weekFromToday;
            } else if ($request->expectation == 2) {
                $expectation = $twoWeeksFromToday;
            } else if ($request->expectation == 3) {
                $expectation = $threeWeeksFromToday;
            } else if ($request->expectation == 4) {
                $expectation = $oneMonthLaterFormatted;
            }

            // Insert production order
            $order = new ProductionOrder();
            $order->selling_id   = $sellingId;
            $order->date         = $date;
            $order->total        = $total;
            $order->type         = $type;
            $order->genre        = $genre;
            $order->customer_id  = isset($customerId) ? $customerId : null;
            $order->paid         = $paid;
            $order->due          = $due;
            $order->expectation  = $expectation;
            $order->account_id   = $accountId;
            $order->user_id      = auth()->id(); // cleaner than auth()->user()->id

            // Determine the stage based on type
            if ($type == 1 || $type == 2) {
                $order->stage = 1;
            } elseif ($type == 3 || $type == 4) {
                $order->stage = 5;
            }

            $order->save();

            //loop through items
            foreach ($items as $item) {
                // You may want to resolve color ID from $item['rangi']
                $color = Product::where('product', $item['rangi'])->first(); // Adjust as needed
                $colorId = $color ? $color->id : null; // Optional fallback if not found

                $productionItem = new ProductionOrderItem();
                $productionItem->production_order_id = $order->id;
                $productionItem->product_id = $item['id'];
                $productionItem->unit_id = $item['unit'];
                $productionItem->unit_cont = $item['cont'];
                $productionItem->size = $item['size'];
                $productionItem->color = $colorId;
                $productionItem->main = $item['main'];
                $productionItem->sub = $item['sub'];
                $productionItem->qty = $item['qty'];
                $productionItem->price = $item['price'];
                $productionItem->amount = $item['amount'];
                $productionItem->save();
            }

            // commit
            DB::commit();

            return redirect()->back()->with("success", "Order places successfully");
        } catch (Exception $e) {
            // rollback
            DB::rollBack();
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function progress($flag)
    {
        # Get location details
        $locationDetails = processLocationDetails();
        $sellings = $locationDetails['sellings'];
        $sellingIds = $sellings->pluck('id')->toArray();

        # Start query
        $orderQuery = ProductionOrder::whereIn('selling_id', $sellingIds)
            ->where('stage', '<=', 6)
            ->orderBy('created_at', 'ASC');

        if ($flag == 2) { # filter by expectation
            $orderQuery->where('expectation', '<', date('Y-m-d'));
        }

        $orders = $orderQuery->with([
            'selling.location',
            'customer',
            'items.product' => function ($query) {
                $query->withTrashed(); // Include soft-deleted products
            },
            'items.unit'
        ])->get();

        return view('production-order-progress', compact('orders', 'flag', 'sellings'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function processOrder($order_id, $stage)
    {
        // Start transaction
        DB::beginTransaction();

        try {
            // Get order details
            $order = ProductionOrder::where('id', $order_id)->first();
            $order->stage = $stage == 1 ? $order->stage - 1 : $order->stage + 1;
            $order->update();

            if ($stage == 1) {
                $message = 'Order reverted on stage';
            } elseif ($order->stage == 7) {
                $message = 'Order completed, Ready for delivery';
            } elseif ($stage == 2) {
                $message = 'Order progressed one stage';
            }

            // Commit transaction
            DB::commit();

            return back()->with('success', $message);
        } catch (\Exception $e) {
            // Rollback transaction
            DB::rollBack();
            return back()->with('error', 'Something went wrong');
        }
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        //
    }

    public function postpondOrder(Request $request, $order_id)
    {
        # Get order By Id from production order table
        $order = ProductionOrder::where('id', $order_id)->first();
        if ($order) {
            $order->expectation = $request->date;
            $order->update();

            return back()->with('success', 'Postpoded successfully');
        } else {
            return back()->with('error', 'Order not found');
        }
    }
    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($order_id)
    {
        # Get order.
        $order = ProductionOrder::find($order_id);

        if ($order) { # if the order is found
            ProductionOrderItem::where('production_order_id', $order->id)->delete(); # Delete the order items
            $order->delete(); # Delete Order
            return back()->with('success', 'Cancelled successfully');
        } else { # If its not found
            return back()->with('error', 'Order not found');
        }
    }

    public function dispatchView()
    {
        # Get location details
        $locationDetails = processLocationDetails();
        $sellings = $locationDetails['sellings'];
        $accounts = $locationDetails['accounts'];
        $sellingIds = $sellings->pluck('id')->toArray();

        # Get orders for dispatch
        $orders = ProductionOrder::whereIn('selling_id', $sellingIds)
            ->where('stage', 7)
            ->orderBy('created_at', 'ASC')
            ->with('selling.location', 'customer', 'items.product', 'items.unit')->get();

        return view('dispatch', compact('orders', 'accounts'));
    }

    public function processDispatch(Request $request, $order_id)
    {
        // Start transaction
        DB::beginTransaction();

        // get transport details
        $trans = $request->transport == 1 ? 'Pikipiki' : "Gari";
        $cost = $request->cost;
        $trip = $request->trip;

        $transDetails = [$trans, $cost, $trip];

        try {
            // update order stage
            $order = ProductionOrder::where('id', $order_id)
                ->with(
                    'items.unit',
                    'selling.location',
                    'customer'
                )->first();

            // get customer
            $customer = Customer::where('id', $order->customer_id)->first();

            if ($order) { // order exists
                $order->stage = 8;
                $order->update();

                // create debt
                if ($order->due > 0) {
                    if ($customer) { // customer exists
                        // Check if the customer has existing debts at the same location and update the debt amount.
                        $existingDebt = Debts::where('customer_id', $order->customer_id)
                            ->where('selling_id', $order->selling_id)
                            ->latest()
                            ->first();

                        $debt = new Debts();
                        $debt->date = date('Y-m-d');
                        $debt->production_order_id = $order->id;
                        $debt->customer_id = $order->customer_id;
                        $debt->amount = $order->total;
                        $debt->discount = 0;
                        $debt->paid = $order->paid;
                        $debt->due = ($existingDebt ? $existingDebt->due : 0) + ($order->total - $order->paid);
                        $debt->selling_id = $order->selling_id;
                        $debt->save();
                    }
                }
            }

            // Commit transaction
            DB::commit();

            # Initiate variables
            $sellingId = $order->selling_id;
            $profileData = BusinessProfile::where("selling_id", $sellingId)->first();

            return view('dispatch-memo', compact('order', 'customer', 'profileData', 'transDetails'));
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'Something went wrong');
        }
    }

    public function orderPayment(Request $request, $order_id)
    {
        DB::beginTransaction();

        try {
            // Get order By Id from production order table
            $order = ProductionOrder::where('id', $order_id)->first();

            // Format the paid amount.
            $paid = (int)str_replace(',', '', $request->paid);

            if ($order) { // order exists
                $order->paid += $paid;
                $order->due -= $paid;
                $order->user_id = auth()->user()->id;
                $order->update();

                // Get account details
                $account = Account::find($request->account)->first();

                if ($account) { // account exists
                    // create transaction
                    $transaction = new Transaction();
                    $transaction->date = $request->date;
                    $transaction->reason = 'Production Order Payment';
                    $transaction->status = 'In';
                    $transaction->amount = $paid;
                    $transaction->before = $account->balance;
                    $transaction->after = $account->balance + $paid;
                    $transaction->account_id = $account->id;
                    $transaction->user_id = auth()->user()->id;
                    $transaction->selling_id = $order->selling_id;
                    $transaction->save();

                    $account->balance += $paid;
                    $account->update();
                }
                // Commit transaction
                DB::commit();

                return back()->with('success', 'Payment posted successfully');
            } else { // order does not exist
                return back()->with('error', 'Order not found');
            }
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'Something went wrong');
        }
    }

    public function processStockOrder(Request $request, $orderId)
    {
        if (!$request->area) {
            return redirect()->back()->with('error', 'Location is required.');
        }

        DB::beginTransaction();

        try {
            $destinationId = $request->area;

            $order = ProductionOrder::with('selling.location')->find($orderId);

            if (!$order) {
                return back()->with('error', 'Order not found.');
            }

            $orderItems = ProductionOrderItem::where('production_order_id', $orderId)->get();

            foreach ($orderItems as $item) {
                $qtyToAdd = $item->qty * $item->unit_cont;

                $sourceProduct = Product::where('id', $item->product_id)
                    ->where('location_id', $order->selling->location->id)
                    ->first();

                if (!$sourceProduct) {
                    return back()->with('error', 'Product not found.');
                }

                // Check for existing product in destination
                $destinationProduct = Product::where('id', $sourceProduct->id)
                    ->where('location_id', $destinationId)
                    ->first();

                if ($destinationProduct) {
                    // Update existing stock
                    $stock = Stock::where('product_id', $destinationProduct->id)
                        ->where('location_id', $destinationId)
                        ->firstOrFail();

                    StockMovement::create([
                        'user_id' => auth()->id(),
                        'date' => now()->toDateString(),
                        'product_id' => $destinationProduct->id,
                        'reason' => 'Production Order',
                        'status' => 'In',
                        'qty' => $qtyToAdd,
                        'before' => $stock->qty,
                        'after' => $stock->qty + $qtyToAdd,
                        'location_id' => $destinationId,
                    ]);

                    // Update stock quantity
                    $stock->increment('qty', $qtyToAdd);
                } else {
                    // Clone product to destination
                    $newProduct = $sourceProduct->replicate();
                    $newProduct->location_id = $destinationId;
                    $newProduct->save();

                    // Clone units
                    $units = UnitAssigned::where('product_id', $sourceProduct->id)
                        ->where('location_id', $sourceProduct->location_id)
                        ->get();

                    foreach ($units as $unit) {
                        UnitAssigned::create([
                            'product_id' => $newProduct->id,
                            'unit_id' => $unit->unit_id,
                            'unit_cont' => $unit->unit_cont,
                            'selling' => $unit->selling,
                            'location_id' => $destinationId,
                        ]);
                    }

                    // Clone stock
                    $oldStock = Stock::where('product_id', $sourceProduct->id)
                        ->where('location_id', $sourceProduct->location_id)
                        ->first();

                    $newStock = new Stock();
                    $newStock->product_id = $newProduct->id;
                    $newStock->location_id = $destinationId;
                    $newStock->unit_id = $oldStock->unit_id;
                    $newStock->min_cont = $oldStock->min_cont;
                    $newStock->min_qty = $oldStock->min_qty;
                    $newStock->qty = $qtyToAdd;
                    $newStock->exp = $oldStock->exp;
                    $newStock->exp_not = $oldStock->exp_not;
                    $newStock->save();

                    // Record movement
                    StockMovement::create([
                        'user_id' => auth()->id(),
                        'date' => now()->toDateString(),
                        'product_id' => $newProduct->id,
                        'reason' => 'Production Order',
                        'status' => 'In',
                        'qty' => $qtyToAdd,
                        'before' => 0,
                        'after' => $qtyToAdd,
                        'location_id' => $destinationId,
                    ]);
                }
            }

            // Mark order as completed
            $order->update(['stage' => 8]);

            DB::commit();

            return back()->with('success', 'Order completed successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'Something went wrong: ' . $e->getMessage());
        }
    }

    public function report(Request $request)
    {
        // Initialize variables
        $type = $request->type;
        $duration = $request->duration;
        $genre = $request->genre ? $request->genre : 3;
        $currentDate = date('Y-m-d');
        $start = $request->start;
        $end = $request->end;
        $orders = [];
        $items = [];

        if ($request->has('genre') && $request->has('duration')) {
            // Initialize query
            $query = ProductionOrder::query()->with('user', 'customer', 'items');

            if ($genre == 1) {
                $query->where('genre', 1)->whereNotNull('customer_id');
            } elseif ($genre == 2) {
                $query->where('genre', 2)->where('stage', 8);
            }

            if ($duration == 1) { // Today
                $query->whereDate('date', $currentDate);
            } elseif ($duration == 2) { // Set date
                $query->whereDate('date', $start);
            } elseif ($duration == 3) { // Set duration
                $query->whereBetween('date', [$start, $end]);
            }

            if ($type == 2) { // dispached
                $query->where('stage', 8);
            }

            $orders = $query->get(); // get production orders

            if ($genre == 2) {
                $orderIds = $orders->pluck('id')->toArray();

                // Get production order items
                $items = ProductionOrderItem::whereIn('production_order_id', $orderIds)
                    ->with('unit', 'product')
                    ->selectRaw('product_id, sum(qty) as total_qty, unit_id, price, created_at')
                    ->groupBy('product_id')
                    ->get();
            }
        }

        return view('production-report', compact('orders', 'genre', 'items'));
    }
}
