<?php

namespace App\Http\Controllers;

use App\Models\PurchaseReturn;
use App\Models\PurchaseReturnItem;
use App\Models\Account;
use App\Models\DraftPurchase;
use App\Models\Location;
use App\Models\Supplier;
use App\Models\Transaction;
use App\Models\Product;
use App\Models\Purchase;
use App\Models\PurchasedItem;
use App\Models\PurchaseOrder;
use App\Models\Selling;
use App\Models\Stock;
use App\Models\StockMovement;
use App\Models\SupplierDebts;
use App\Models\UnitAssigned;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class PurchaseController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index($id, $status)
    {
        try {
            # Get location details
            $locationDetails = processLocationDetails();
            $sellingCount = $locationDetails['sellingCount'];
            $sellings = $locationDetails['sellings'];
            $suppliers = $locationDetails['suppliers'];

            if ($status == 'purchase-order') {
                $purchaseOrderId = $id;

                // Retrieve purchase order data and items from session if available
                $purchaseOrderData = PurchaseOrder::with([
                    'purchaseOrderItems.product.unit.unit',
                    'supplier',
                    'selling',
                    'user'
                ])->find($purchaseOrderId);

                $purchaseOrderItems = $purchaseOrderData->purchaseOrderItems ?? [];

                // Add the two variables to their respective sessions
                session(['purchaseOrderData' => $purchaseOrderData]);
                session(['purchaseOrderItems' => $purchaseOrderItems]);
            } else {
                session()->forget(['purchaseOrderData', 'purchaseOrderItems', 'supplierData']);
            }

            return view("purchases", compact("sellings", "sellingCount", "suppliers"));
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function returnView()
    {
        return response()->json(['message' => 'This functionality is still under maintenance.']);

        try {
            # Get location details
            $locationDetails = processLocationDetails();
            $sellingCount = $locationDetails['sellingCount'];
            $sellings = $locationDetails['sellings'];
            $suppliers = $locationDetails['suppliers'];


            return view("purchase-return", compact("sellings", "sellingCount", "suppliers"));
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        try {
            DB::beginTransaction();

            // Sanitize numeric inputs
            $total = (int) str_replace(',', '', $request->total);
            $net = (int) str_replace(',', '', $request->net);
            $paid = (int) str_replace(',', '', $request->paid);
            $due = (int) str_replace(',', '', $request->due);

            $flag = 'in';
            $originLocation = $request->sehemu;
            $storageLocation = $request->storage;
            $reason = "Purchase";
            $sellingId = Selling::where('location_id', $originLocation)->value('id');

            // Supplier handling
            $supplierId = null;
            if ($request->supplier) {
                $supplierData = Supplier::firstOrCreate(
                    ['name' => $request->supplier, 'selling_id' => $sellingId],
                    [
                        'address' => ucwords($request->address),
                        'phone' => $request->phone ?? "Unknown",
                        'opening_balance' => 0,
                        'advance' => 0
                    ]
                );
                $supplierId = $supplierData->id;
            }

            // Create purchase
            $purchase = Purchase::create([
                'user_id' => auth()->id(),
                'date' => $request->date,
                'total' => $total,
                'net' => $net,
                'paid' => $paid,
                'due' => $due,
                'account_id' => $request->account,
                'supplier_id' => $supplierId,
                'invoice' => $request->invoice,
                'selling_id' => $sellingId,
                'storage_id' => $storageLocation,
                'status' => $request->type === 'cash' ? 'cash' : 'credit',
            ]);

            // Handle credit debts (kept)
            if ($request->type === 'credit' && $supplierId) {
                $existingDebt = SupplierDebts::where('supplier_id', $supplierId)
                    ->where('selling_id', $sellingId)
                    ->latest()
                    ->first();

                $dueAmount = $existingDebt ? $existingDebt->due + $due : $due;

                SupplierDebts::create([
                    'date' => $request->date,
                    'purchase_id' => $purchase->id,
                    'supplier_id' => $supplierId,
                    'amount' => $total,
                    'paid' => $paid,
                    'due' => $dueAmount,
                    'selling_id' => $sellingId,
                ]);
            }

            # Process Cash Movement..
            if ($paid > 0) {
                if ($request->account === 'advance' && $supplierId) {
                    Supplier::find($supplierId)->decrement('advance', $paid);
                } else {
                    $account = Account::find($request->account);
                    recordTransaction(
                        $request->date,
                        'Purchase',
                        'Out',
                        $paid,
                        $account->id,
                        auth()->id(),
                    );
                }
            }

            // Loop through purchased items
            for ($i = 0; $i < count($request->id); $i++) {
                $buying = floatval(str_replace(',', '', $request->buying[$i]));
                $amount = floatval(str_replace(',', '', $request->amount[$i]));
                $qty = (int) $request->qty[$i];
                $cont = (int) $request->cont[$i];
                $batchNo = $request->batch[$i] ?? null;
                $expDate = $request->exp[$i] ?? null;

                // Insert or update PurchasedItem
                $itemData = [
                    'qty' => DB::raw("qty + $qty"),
                    'amount' => DB::raw("amount + $amount"),
                    'buying' => $buying,
                    'cont' => $cont,
                ];

                if ($batchNo) {
                    $itemData['batch_no'] = $batchNo;
                    $itemData['exp_date'] = $expDate;
                }

                $item = PurchasedItem::updateOrCreate(
                    [
                        'purchase_id' => $purchase->id,
                        'product_id' => $request->id[$i],
                        'unit_id' => $request->unit[$i],
                    ] + ($batchNo ? ['batch_no' => $batchNo] : []),
                    $itemData
                );

                $existingProduct = Product::where('id', $request->id[$i])
                    ->where('location_id', $storageLocation)
                    ->first();

                if ($existingProduct) {

                    $stockQuery = [
                        'product_id' => $existingProduct->id,
                        'location_id' => $storageLocation
                    ];

                    if ($batchNo) {
                        $stockQuery['batch_no'] = $batchNo;
                        $stockQuery['exp_date'] = $expDate;
                    }

                    $stock = Stock::firstOrCreate($stockQuery, [
                        'unit_id' => $request->unit[$i],
                        'qty' => 0,
                        'min_qty' => 0,
                        'min_cont' => 0,
                    ]);

                    recordStockMovement(
                        $request->date,
                        auth()->id(),
                        $existingProduct->id,
                        $storageLocation,
                        $reason,
                        $flag,
                        $qty * $cont,
                        $batchNo,
                        $expDate
                    );

                    $existingProduct->buying = $buying / $cont;
                    $existingProduct->update();

                    $priceKey = "sell" . $existingProduct->id;
                    $prices = $request->input($priceKey, []);
                    $units = UnitAssigned::where('product_id', $existingProduct->id)
                        ->where('location_id', $storageLocation)->get();

                    foreach ($units as $index => $unit) {
                        $sell = isset($prices[$index]) ? str_replace(',', '', $prices[$index]) : 0;
                        $unit->selling = $sell;
                        $unit->save();
                    }
                } else {

                    $oldProduct = Product::find($request->id[$i]);
                    $newProduct = $oldProduct->replicate();
                    $newProduct->location_id = $storageLocation;
                    $newProduct->buying = $buying / $cont;
                    $newProduct->save();

                    $oldUnits = UnitAssigned::where('product_id', $request->id[$i])
                        ->where('location_id', $originLocation)
                        ->get();

                    $priceKey = "sell" . $oldProduct->id;
                    $prices = $request->input($priceKey, []);

                    foreach ($oldUnits as $index => $oldUnit) {
                        $selling = isset($prices[$index]) ? str_replace(',', '', $prices[$index]) : 0;

                        UnitAssigned::create([
                            'product_id' => $newProduct->id,
                            'unit_id' => $oldUnit->unit_id,
                            'unit_cont' => $oldUnit->unit_cont,
                            'selling' => $selling,
                            'location_id' => $storageLocation
                        ]);
                    }

                    $newStockData = [
                        'product_id' => $newProduct->id,
                        'location_id' => $storageLocation,
                        'unit_id' => $request->unit[$i],
                        'qty' => $qty * $cont,
                        'min_qty' => 2,
                        'min_cont' => 1,
                    ];

                    if ($batchNo) {
                        $newStockData['batch_no'] = $batchNo;
                        $newStockData['exp_date'] = $expDate;
                    }

                    $newStock = Stock::create($newStockData);

                    recordStockMovement(
                        $request->date,
                        auth()->id(),
                        $newProduct->id,
                        $storageLocation,
                        $reason,
                        $flag,
                        $qty * $cont,
                        $batchNo,
                        $expDate
                    );
                }
            }

            DB::commit();

            foreach (['purchaseOrderData', 'purchaseOrderItems', 'supplierData'] as $key) {
                session()->forget($key);
            }

            return redirect()->route('purchase.view', ['id' => 0, 'status' => 'purchase'])
                ->with('success', 'Posted successfully.');
        } catch (Exception $e) {
            DB::rollBack();
            Log::error($e->getMessage());
            return response()->json(['error' => $e->getMessage()], 500);
        }
    }

    public function payInvoice(Request $request, $id)
    {
        try {
            // Begin the database transaction
            DB::beginTransaction();

            // Retrieve the purchase by its ID
            $purchase = Purchase::where("id", $request->id)->first();

            // Update the due and paid amounts of the purchase
            $purchase->due -= $request->paid;
            $purchase->paid += $request->paid;

            // Check if the due amount is now zero
            if ($purchase->due == 0) {
                // If the due amount is zero, update the status, duedate, and notification fields
                $purchase->status = "cash";
            }

            // Update the purchase in the database
            $purchase->update();

            // Retrieve the account related to the payment
            $account = Account::find($request->account);

            // Prepare the data for the transaction
            $transactionData = [
                'user_id' => auth()->user()->id,
                'date' => $request->date,
                'reason' => "Paid pending purchase",
                'status' => "Out",
                'amount' => $request->paid,
                'before' => $account->balance,
                'after' => $account->balance - $request->paid,
                'account_id' => $account->id,
                'location_id' => $purchase->sehemu,
            ];

            // Create a new transaction with the prepared data
            Transaction::create($transactionData);

            // Decrement the balance of the account by the paid amount
            $account->decrement('balance', $request->paid);

            // Commit the database transaction
            DB::commit();

            // Redirect back with a success message
            return back()->with("success", "Payment successfully posted!!");
        } catch (Exception $e) {
            // An error occurred, rollback the transaction
            DB::rollBack();
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function paySupplierDebtHistory(Request $request, $supplierId)
    {
        $request->validate([
            'account' => 'required',
            'paid' => 'required|min:1',
            'date' => 'required|date',
        ]);

        $paid = (int)str_replace(',', '', $request->paid);

        DB::beginTransaction();
        try {
            $supplier = Supplier::find($supplierId);
            if (!$supplier) {
                return back()->with('error', 'Supplier not found.');
            }

            $sellingId = $supplier->selling_id;
            $accountId = $request->account;

            // 1️⃣ Create payment in purchase system (existing method)
            $purchase = new Purchase();
            $purchase->payDebt($supplierId, $paid, $sellingId);

            // 2️⃣ Get latest debt
            $currentDebt = SupplierDebts::where("supplier_id", $supplierId)
                ->where("selling_id", $sellingId)
                ->latest()
                ->first();

            if (!$currentDebt) {
                return back()->with('error', 'Debt not found.');
            }

            // 3️⃣ Record supplier debt payment
            $debt = new SupplierDebts();
            $debt->date = $request->date;
            $debt->supplier_id = $supplierId;
            $debt->paid = $paid;
            $debt->account_id = $accountId;
            $debt->due = max(0, $currentDebt->due - $paid);
            $debt->selling_id = $sellingId;
            $debt->save();

            // 4️⃣ Process account transaction using helper
            $fakeRequest = new \Illuminate\Http\Request([
                'date' => $request->date,
                'account' => $accountId,
                'paid' => $paid,
            ]);

            processAccountTransaction($fakeRequest, $debt, 'supplier-payment');

            // 5️⃣ Update all remaining debts for the supplier
            updateSupplierDebts($sellingId, $supplierId);

            DB::commit();
            return back()->with("success", "Payment successfully posted.");
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error($e->getMessage());
            return back()->with('error', 'Something went wrong.');
        }
    }

    public function purchaseReturn(Request $request)
    {
        try {
            // Begin the database transaction
            DB::beginTransaction();

            // Create a new PurchaseReturn object and save it
            $return = new PurchaseReturn($request->only(['date', 'total', 'account', 'supplier', 'area']));
            $return->user_id = auth()->user()->id;
            $return->save();

            $items = [];            // Array to store PurchaseReturnItem data
            $stockItems = [];       // Array to store Stock data for existing products
            $movementItems = [];    // Array to store StockMovement data for existing products

            // Iterate through the items in the request
            for ($i = 0; $i < count($request->id); ++$i) {
                // Create an item array and add it to the items array
                $items[] = [
                    'return_id' => $return->id,
                    'product_id' => $request->id[$i],
                    'unit_id' => $request->unit[$i],
                    'cont' => $request->cont[$i],
                    'price' => $request->buying[$i],
                    'qty' => $request->qty[$i],
                    'amount' => $request->amount[$i],
                    'created_at' => now(),
                    'updated_at' => now(),
                ];

                // Check if stock exists for the product and location
                if (Stock::where("product_id", $request->id[$i])
                    ->where("location_id", $request->area)
                    ->exists()
                ) {
                    // Retrieve the stock and update it
                    $stock = Stock::where("product_id", $request->id[$i])
                        ->where("location_id", $request->area)
                        ->first();
                    // Create a movement item and add it to the movement items array
                    $movementItems[] = [
                        'date' => $request->date,
                        'user_id' => auth()->user()->id,
                        'product_id' => $request->id[$i],
                        'reason' => "Purchase Return",
                        'status' => "Out",
                        'qty' => $request->qty[$i] * $request->cont[$i],
                        'before' => $stock->qty,
                        'after' => $stock->qty - ($request->qty[$i] * $request->cont[$i]),
                        'location_id' => $request->area,
                        'created_at' => now(),
                        'updated_at' => now(),
                    ];

                    $stock->qty -= $request->qty[$i] * $request->cont[$i];
                    $stock->save();
                } else {
                    // Get the old product from products table
                    $oldProduct = Product::where("id", $request->id[$i])
                        ->where("location_id", $request->location)
                        ->first();
                    // Create new product in the respective location with old products details.
                    $newProduct = new Product();
                    $newProduct->barcode = $oldProduct->barcode;
                    $newProduct->image = $oldProduct->image;
                    $newProduct->barcode = $oldProduct->barcode;
                    $newProduct->category_id = $oldProduct->category_id;
                    $newProduct->buying = $oldProduct->buying;
                    $newProduct->supplier_id = $oldProduct->supplier_id;
                    $newProduct->location_id = $request->area;
                    $newProduct->save();

                    // Get the old product units
                    $oldUnits = UnitAssigned::where("Product_id", $request->id[$i])
                        ->where("location_id", $request->location)
                        ->get();
                    // Loop through the old units
                    foreach ($oldUnits as $oldUnit) {
                        //create new units in  units assigned table.
                        $newUnit = new UnitAssigned();
                        $newUnit->product_id = $oldUnit->product_id;
                        $newUnit->unit_id = $oldUnit->unit_id;
                        $newUnit->unit_cont = $oldUnit->unit_cont;
                        $newUnit->selling = $oldUnit->selling;
                        $newUnit->location_id = $request->area;
                        $newUnit->save();
                    }

                    // Get the old product from products table.
                    $oldStock = stock::where("product_id", $request->id[$i])
                        ->where("location_id", $request->location)
                        ->first();

                    // Create a stock item and add it to the stock items array
                    $stockItems[] = [
                        'product_id' => $request->id[$i],
                        'location_id' => $request->area,
                        'unit_id' => $oldStock->unit_id,
                        'min_cont' => $oldStock->min_cont,
                        'min_qty' => $oldStock->min_qty,
                        'qty' => $request->qty[$i] * $request->cont[$i],
                        'exp' => $oldStock->exp,
                        'exp_not' => $oldStock->exp_not,
                        'created_at' => now(),
                        'updated_at' => now(),
                    ];

                    // Create a movement item and add it to the movement items array
                    $movementItems[] = [
                        'date' => $request->date,
                        'user_id' => auth()->user()->id,
                        'product_id' => $request->id[$i],
                        'reason' => "Purchase return",
                        'status' => "Out",
                        'qty' => $request->qty[$i] * $request->cont[$i],
                        'before' => 0,
                        'after' => $request->qty[$i] * $request->cont[$i],
                        'location_id' => $request->area,
                        'created_at' => now(),
                        'updated_at' => now(),
                    ];
                }
            }

            // Bulk insert the PurchaseReturnItem data
            PurchaseReturnItem::insert($items);

            // Bulk insert the Stock data for new products
            Stock::insert($stockItems);

            // Bulk insert the StockMovement data
            StockMovement::insert($movementItems);

            // Increment the balance of the account
            Account::find($request->account)->increment('balance', $request->total);

            // Retrieve the account
            $account = Account::find($request->account);

            // Create a new transaction and save it
            $transaction = new Transaction();
            $transaction->date = $request->date;
            $transaction->user_id = auth()->user()->id;
            $transaction->reason = "Purchase return";
            $transaction->status = "In";
            $transaction->amount = $request->total;
            $transaction->before = $account->balance - $request->total;
            $transaction->after = $account->balance;
            $transaction->account_id = $account->id;
            $transaction->location_id = $request->area;
            $transaction->save();

            // Commit the database transaction
            DB::commit();

            // Redirect to the purchase return view with a success message
            return redirect()->route('purchase.return.view')->with("success", "Posted successfully!!");
        } catch (Exception $e) {
            // An error occurred, rollback the transaction
            DB::rollBack();
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        //
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */

    public function bulkDelete(Request $request)
    {
        try {
            DB::beginTransaction();

            $ids = $request->ids;
            $purchases = Purchase::whereIn('id', $ids)->get();

            foreach ($purchases as $purchase) {
                $locationId = Selling::where('id', $purchase->selling_id)->value('location_id');

                // 1. Handle supplier debts for credit purchases
                if ($purchase->status === 'credit' && $purchase->supplier_id) {
                    SupplierDebts::where('purchase_id', $purchase->id)->delete();

                    $supplierDebts = SupplierDebts::where('supplier_id', $purchase->supplier_id)
                        ->where('selling_id', $purchase->selling_id)
                        ->orderBy('date', 'asc')
                        ->get();

                    $openingBalance = 0;
                    foreach ($supplierDebts as $debt) {
                        $debt->opening_balance = $openingBalance;
                        $debt->due = $debt->opening_balance + $debt->amount - $debt->paid - ($debt->discount ?? 0);
                        $debt->save();
                        $openingBalance = $debt->due;
                    }
                }

                // 2. Handle purchased items and batch-aware stock reversal
                $items = PurchasedItem::where("purchase_id", $purchase->id)->get();
                foreach ($items as $item) {
                    // Find the stock for the exact batch (if exists)
                    $stock = Stock::where('product_id', $item->product_id)
                        ->where('location_id', $locationId)
                        ->when($item->batch_no, function ($query) use ($item) {
                            $query->where('batch_no', $item->batch_no);
                        })
                        ->first();

                    if ($stock) {
                        // Reverse stock movement
                        recordStockMovement(
                            date('Y-m-d'),
                            auth()->id(),
                            $item->product_id,
                            $locationId,
                            'Purchase deleted',
                            'Out',
                            $item->qty * $item->cont,
                            $item->batch_no,
                            $item->exp_date
                        );
                    } else {
                        // Optional: log or handle missing batch
                        Log::warning("Stock batch missing for product {$item->product_id} at location {$locationId}");
                    }

                    // Delete the purchased item
                    $item->delete();
                }

                // 3. Reverse account transactions if any
                if ($purchase->paid > 0 && $purchase->account_id) {
                    $account = Account::find($purchase->account_id);
                    if ($account) {
                        recordTransaction(
                            date('Y-m-d'),             // date
                            'Purchase deleted',         // reason
                            'In',                       // status (money returning)
                            $purchase->paid,            // amount
                            $account->id,               // account_id
                            auth()->id(),               // user_id
                        );
                    }
                }

                // 4. Finally, delete the purchase record
                $purchase->delete();
            }

            DB::commit();
            return response()->json('success');
        } catch (Exception $e) {
            DB::rollBack();
            Log::error($e->getMessage());
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }


    public function itemsMissingProductId()
    {
        try {
            // Begin the database transaction
            DB::beginTransaction();

            // Get all items from the sold_items table where product_id column is null.
            $purchased_items = PurchasedItem::whereNull('product_id')->get();

            // Loop through the items.
            foreach ($purchased_items as $item) {
                // Find the product based on the item's product name.
                $product = Product::where('product', $item->product)->first();

                if ($product) {
                    // Update the product_id column in the sold_items table.
                    $item->product_id = $product->id;
                    $item->save();
                }
            }

            // Commit the database transaction
            DB::commit();

            return response()->json(['message' => 'All purchased items missing product_id have been updated successfully!!']);
        } catch (Exception $e) {
            // An error occurred, rollback the transaction
            DB::rollBack();
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function purchaseOrderView()
    {
        try {

            $query = PurchaseOrder::query();

            if (auth()->user()->selling_location_id != "all") {
                $query->where("location_id", auth()->user()->selling_location_id)->get();
            }
            $purchaseOrders = $query->with("location", "customer", "user")->get();

            return view("profoma-invoice-view", compact("purchaseOrders"));
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function cancelPurchase()
    {
        // Clear the session data only if they exist
        foreach (['purchaseOrderData', 'purchaseOrderItems', 'supplierData'] as $key) {
            if (session()->has($key)) {
                session()->forget($key);
            }
        }

        return redirect()->route('purchase.view', ['id' => 0, 'status' => 'purchase'])->with('success', 'Purchase cancelled.');
    }
}
