<?php

namespace App\Http\Controllers;

use App\Models\Account;
use App\Models\Asset;
use App\Models\Booking;
use App\Models\BusinessProfile;
use App\Models\Customer;
use App\Models\Damage;
use App\Models\Debts;
use App\Models\License;
use App\Models\LoadVehicle;
use App\Models\Location;
use Illuminate\Http\Request;
use App\Models\PostExpense;
use App\Models\Product;
use App\Models\ProductCategory;
use App\Models\ProductionOrder;
use App\Models\ProductionOrderItem;
use App\Models\Purchase;
use App\Models\PurchasedItem;
use App\Models\Sale;
use App\Models\SaleReturn;
use App\Models\Selling;
use App\Models\SoldItem;
use App\Models\Stock;
use App\Models\StockMovement;
use App\Models\StockTacking;
use App\Models\StockTransfer;
use App\Models\Supplier;
use App\Models\SupplierDebts;
use App\Models\Transaction;
use App\Models\Waiter;
use Carbon\Carbon;
use Exception;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use function updateCustomerDebts; // Import the function

class ReportController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    public function expenseView()
    {
        try {
            # Get location details
            $locationDetails = processLocationDetails();
            $sellingCount = $locationDetails['sellingCount'];
            $sellings = $locationDetails['sellings'];
            $accounts = $locationDetails['accounts'];

            $duration = "";
            $currentDate = "";
            $start = "";
            $end = "";
            $type = "";
            $total = 0;
            $results = [];
            return view("expense-report", compact(
                "currentDate",
                "start",
                "end",
                "type",
                "results",
                "total",
                "sellings",
                "sellingCount",
                "accounts",
                "duration",
            ));
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function expense(Request $request)
    {
        try {
            // Get location details
            $locationDetails = processLocationDetails();
            $sellingCount = $locationDetails['sellingCount'];
            $sellings = $locationDetails['sellings'];
            $accounts = $locationDetails['accounts'];

            $currentDate = date('Y-m-d');
            $start = $request->start ?? null;
            $end = $request->end ?? null;
            $type = $request->type ?? 'today';
            $seletedLocation = $request->location ?? 'all';

            // Base query
            $baseQuery = PostExpense::with(['user', 'selling', 'category'])
                ->when($seletedLocation !== 'all', function ($q) use ($seletedLocation) {
                    $q->where('selling_id', $seletedLocation);
                });

            if ($type === 'today') {
                $results = (clone $baseQuery)
                    ->where('date', $currentDate)
                    ->get();

                $total = (clone $baseQuery)
                    ->where('date', $currentDate)
                    ->sum('sum');
            } elseif ($type === 'set date') {
                $results = (clone $baseQuery)
                    ->where('date', $start)
                    ->get();

                $total = (clone $baseQuery)
                    ->where('date', $start)
                    ->sum('sum');
            } else {
                $results = (clone $baseQuery)
                    ->whereBetween('date', [$start, $end])
                    ->get();

                $total = (clone $baseQuery)
                    ->whereBetween('date', [$start, $end])
                    ->sum('sum');
            }

            return view('expense-report', compact(
                'currentDate',
                'start',
                'end',
                'type',
                'results',
                'total',
                'seletedLocation',
                'sellings',
                'sellingCount',
                'accounts'
            ));
        } catch (\Exception $e) {
            Log::error($e->getMessage());
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }


    public function stockTransfer(Request $request)
    {
        try {
            # Get location details
            $locationDetails = processLocationDetails();
            $sellingCount = $locationDetails['sellingCount'];
            $sellings = $locationDetails['sellings'];

            // Get the current date
            $currentDate = date('Y-m-d');

            // Get the request parameters
            $start = $request->start;
            $end = $request->end;
            $type = $request->type;
            $location_id = $request->area;

            // Initialize the results array
            $results = [];

            // Create a query builder instance
            $query = StockTransfer::where("location_id", $location_id);

            // Check the value of the type parameter
            if ($type == "today") {
                // Add a condition to filter by the current date
                $query->whereDate("date", $currentDate);
            } elseif ($type == "set date") {
                // Add a condition to filter by the start date
                $query->whereDate("date", $start);
            } else {
                // Add a condition to filter by a date range
                $query->whereBetween("date", [$start, $end]);
            }

            // Execute the query and get the results
            $results = $query->with("product.unit", "unit", "location", "user")->get();

            // Return the results to the stock-transfer-report view
            return view("stock-transfer-report", compact("currentDate", "start", "end", "type", "results", "sellings", "sellingCount"));
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function damageStock(Request $request)
    {
        try {
            # Get location details
            $locationDetails = processLocationDetails();
            $sellingCount = $locationDetails['sellingCount'];
            $sellings = $locationDetails['sellings'];

            // Get the current date
            $currentDate = date('Y-m-d');

            // Get the start and end dates from the request
            $start = $request->start;
            $end = $request->end;

            // Get the type and location from the request
            $type = $request->type;
            $location_id = $request->area;

            // Initialize an empty array to store the results
            $damageQuery = Damage::where("location_id", $location_id);

            // Retrieve damage records based on the selected type and location
            if ($type == "today") {
                // Retrieve damage records for the current date and specific location
                $damageQuery->where("date", $currentDate);
            } elseif ($type == "set date") {
                // Retrieve damage records for a specific date and location
                $damageQuery->where("date", $start);
            } else {
                // Retrieve damage records within a date range and specific location
                $damageQuery->whereBetween("date", [$start, $end]);
            }

            $results = $damageQuery->with("location", "product", "user", "unit")->get();

            // Return the view with the necessary data
            return view("damage-stock-report", compact("currentDate", "start", "end", "type", "results", "sellings", "sellingCount"));
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function customerDebts(Request $request)
    {
        try {
            # Get location details
            $locationDetails = processLocationDetails();
            $authUserSellingId = $locationDetails['userSellingLocation'];

            // Check if the user's location is set to "all"
            if ($authUserSellingId == "all") {
                // Retrieve the latest debts for each customer
                $latestDebts = DB::table('debts')
                    ->select('customer_id', DB::raw('MAX(created_at) as latest_entry'))
                    //->where('due', '>', 0)
                    ->groupBy('customer_id')
                    ->latest()
                    ->get();

                $customerIds = $latestDebts->pluck('customer_id')->toArray();

                // Retrieve the specific debts based on the latest entry for each customer
                $debts = Debts::whereIn('customer_id', $customerIds)
                    ->whereIn('created_at', $latestDebts->pluck('latest_entry')->toArray())
                    ->with('selling.location')
                    ->get();
            } else {
                // Retrieve the latest debts for each customer in the specified location
                $latestDebts = DB::table('debts')
                    ->select('customer_id', DB::raw('MAX(created_at) as latest_entry'))
                    //->where('due', '>', 0)
                    ->where('selling_id', $authUserSellingId)
                    ->groupBy('customer_id')
                    ->latest()
                    ->get();

                $customerIds = $latestDebts->pluck('customer_id')->toArray();

                // Retrieve the specific debts based on the latest entry for each customer in the specified location
                $debts = Debts::whereIn('customer_id', $customerIds)
                    ->whereIn('created_at', $latestDebts->pluck('latest_entry')->toArray())
                    ->with('selling.location')
                    ->get();
            }

            // Initiate variable
            $sum = 0;
            $licenseData = License::select('tra')->first()->value('tra');
            $tra = null;

            if ($licenseData != null && auth()->user()->name == 'Admin') {
                $tra = $licenseData / 100;
            }

            // Pass the retrieved data to the "debts-report" view
            return view("customer-debts-report", compact(
                "debts",
                "sum",
                "tra"
            ));
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function clearedCustomerDebts(Request $request)
    {
        try {
            # Get location details
            $locationDetails = processLocationDetails();
            $authUserSellingId = $locationDetails['userSellingLocation'];

            // Retrieve the latest debts for each customer where due is 0
            $latestDebts = Debts::select('customer_id')
                ->selectRaw('MAX(created_at) as latest_entry');
            // ->where('due', 0);

            if ($authUserSellingId != "all") {
                $latestDebts->where('selling_id',  $authUserSellingId);
            }

            $latestDebts = $latestDebts->groupBy('customer_id')
                ->latest()
                ->get();

            $customerIds = $latestDebts->pluck('customer_id')->toArray();

            // Retrieve the specific debts based on the latest entry for each customer where due is 0
            $debts = Debts::whereIn('customer_id', $customerIds)
                ->whereIn('created_at', $latestDebts->pluck('latest_entry')->toArray())
                // ->where('due', 0)
                ->get();

            // Calculate the sum of the debts (if needed)
            $sum = 0;

            // Pass the retrieved data to the "cleared-customer-debts-report" view
            return view("cleared-customer-debts-report", compact("debts", "sum"));
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function customerStatement(Request $request, $customerId, $sellingId)
    {
        try {
            // Initiate variable.
            $duration = null;
            $start = null;
            $end = null;

            $profile = BusinessProfile::where('selling_id', $sellingId)->with('selling.location')->first();

            # Get location details
            $locationDetails = processLocationDetails();
            $accounts = $locationDetails['accounts'];

            $customer = Customer::find($customerId);

            $latestCustomerDebts = debts::where("customer_id", $customerId)
                ->where("selling_id", $sellingId)
                ->with("selling.location")
                ->latest()
                ->first();

            $debtsQuery = debts::query(); // Start query.

            if ($request->duration === null) {
                $debtsQuery->where("customer_id", $customerId)
                    ->where("selling_id", $sellingId)
                    ->orderBy("created_at", "ASC");
            } else {
                $duration = $request->duration;
                $start = $request->start;
                $end = $request->end;

                if ($request->duration === "today") {
                    $debtsQuery->where("customer_id", $customerId)
                        ->where("date", date('Y-m-d'))
                        ->where("selling_id", $sellingId)
                        ->orderBy("created_at", "ASC");
                } elseif ($request->duration === "set date") {
                    $debtsQuery->where("customer_id", $customerId)
                        ->where("date", $request->start)
                        ->where("selling_id", $sellingId)
                        ->orderBy("created_at", "ASC");
                } elseif ($request->duration === "set duration") {
                    $debtsQuery->where("customer_id", $customerId)
                        ->whereBetween("date", [$request->start, $request->end])
                        ->where("selling_id", $sellingId)
                        ->orderBy("created_at", "ASC");
                }
            }

            $saleQuery = Sale::query();
            $debts = $debtsQuery->with('selling.location')->get();

            // Initiate variables
            $sum = 0;
            $licenseData = License::select('tra')->first()->value('tra');
            $tra = null;

            if ($licenseData != null && auth()->user()->name == 'Admin') {
                $tra = $licenseData / 100;
            }

            return view("customer-debts-history", compact(
                "sum",
                "debts",
                "customer",
                "profile",
                "accounts",
                "latestCustomerDebts",
                "saleQuery",
                "duration",
                "start",
                "end",
                "tra"
            ));
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function cancelCustomerDebtTransaction($debtId)
    {
        if (!$debtId) {
            return back()->with('error', 'No valid debt ID provided.');
        }

        $debtData = Debts::find($debtId);

        if (!$debtData) {
            return back()->with('error', 'No debt data found.');
        }

        DB::transaction(function () use ($debtData) {
            $saleData    = Sale::find($debtData->sale_id);
            $accountData = Account::find($debtData->account_id);
            $userId      = Auth::id();
            $today       = now()->toDateString();
            $sellingId   = $debtData->selling_id;
            $customerId  = $debtData->customer_id;

            if ($saleData) {
                // 1️⃣ Reverse payment transaction if any
                if ($saleData->paid > 0 && $accountData) {
                    recordTransaction(
                        $today,
                        "Cancel Sale",
                        "Out",
                        $saleData->paid,
                        $saleData->account_id,
                        $userId
                    );
                }

                // 2️⃣ Restore stock for sold items
                $soldItems = SoldItem::where('sale_id', $saleData->id)->get();

                foreach ($soldItems as $item) {
                    $qtyToAdd = $item->qty * $item->cont;

                    // Call helper directly
                    recordStockMovement(
                        $today,
                        $userId,
                        $item->product_id,
                        $item->location_id,
                        "Cancel Sale",
                        "In", // ✅ match helper’s expectation
                        $qtyToAdd,
                        $item->batch_no ?? null,
                        $item->exp_date ?? null
                    );

                    $item->delete();
                }

                $saleData->delete();
            } elseif ($accountData && $debtData->paid > 0) {
                // Reverse payment for debt only if no sale exists
                recordTransaction(
                    $today,
                    "Cancel Customer Debt",
                    "Out",
                    $debtData->paid,
                    $debtData->account_id,
                    $userId
                );
            }

            // Delete the debt record
            $debtData->delete();

            // Recalculate customer's remaining debts
            updateCustomerDebts($sellingId, $customerId);
        });

        return redirect()->back()->with('success', 'Transaction cancelled successfully.');
    }

    public function cancelSupplierDebtTransaction($debtId)
    {
        if (!$debtId) {
            return redirect()->back()->with('error', 'No valid debt ID provided.');
        }

        $debtData = SupplierDebts::find($debtId);

        if (!$debtData) {
            return redirect()->back()->with('error', 'No debt data found.');
        }

        DB::transaction(function () use ($debtData) {
            $purchaseId = $debtData->purchase_id;
            $supplierId = $debtData->supplier_id;
            $accountId = $debtData->account_id;
            $sellingId = $debtData->selling_id;

            $purchaseData = Purchase::find($purchaseId);

            $userId = Auth::id();
            $today = now()->toDateString();

            if ($purchaseData) {
                // 1️⃣ Reverse payment if any (use processAccountTransaction)
                if ($purchaseData->paid > 0 && $accountId) {
                    $fakeRequest = new \Illuminate\Http\Request([
                        'date' => $today,
                        'account' => $accountId,
                        'paid' => -$purchaseData->paid, // negative to reverse
                    ]);

                    // reuse processAccountTransaction → will create a transaction + update account
                    processAccountTransaction($fakeRequest, $purchaseData, 'cancel-debt');
                }

                // 2️⃣ Reverse stock for purchased items (use recordStockMovement)
                $purchasedItems = PurchasedItem::where('purchase_id', $purchaseData->id)->get();

                foreach ($purchasedItems as $item) {
                    $qtyToRemove = $item->qty * $item->cont;

                    recordStockMovement(
                        $today,
                        $userId,
                        $item->product_id,
                        $item->location_id,
                        "Cancel Purchase",
                        "Out", // must match helper's expected case
                        $qtyToRemove,
                        $item->batch_no,
                        $item->exp_date
                    );

                    $item->delete();
                }

                $purchaseData->delete();
            } else {
                // No purchase found — just reverse debt payment if any
                if ($accountId && $debtData->paid > 0) {
                    $fakeRequest = new \Illuminate\Http\Request([
                        'date' => $today,
                        'account' => $accountId,
                        'paid' => -$debtData->paid,
                    ]);

                    processAccountTransaction($fakeRequest, (object)['paid' => 0], 'cancel-debt');
                }
            }

            // Delete the debt record
            $debtData->delete();

            // Recalculate remaining debts
            updateSupplierDebts($sellingId, $supplierId);
        });

        return redirect()->route('supplier.debts.report')->with('success', 'Supplier debt transaction cancelled successfully.');
    }



    public function soldItems($saleId)
    {
        try {
            $sale = Sale::find($saleId);
            $items = SoldItem::where("sale_id", $saleId)->get();
            $customer = Customer::find($sale->customer_id);
            $profile = BusinessProfile::where('selling_id', $sale->selling_id)->with('selling.location')->first();

            # Get location details
            $locationDetails = processLocationDetails();
            $accounts = $locationDetails['accounts'];

            $licenseData = License::select('tra')->first()->value('tra');
            $tra = null;

            if ($licenseData != null && auth()->user()->name == 'Admin') {
                $tra = $licenseData / 100;
            }

            return view("sold-items", compact("items", "sale", "profile", "customer", "tra", "accounts"));
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function orderItems($orderId)
    {
        try {
            $order = ProductionOrder::find($orderId);
            $items = ProductionOrderItem::where("production_order_id", $orderId)->get();
            $customer = Customer::find($order->customer_id);
            $profile = BusinessProfile::select(
                "name",
                "address",
                "phone",
                "email",
                "logo",
                "slogan"
            )->first();
            return view("order-items", compact("items", "order", "profile", "customer"));
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }
    // public function supplierDebts(Request $request)
    // {
    //     try {
    //         # Get location details
    //         $locationDetails = processLocationDetails();
    //         $authUserSellingId = $locationDetails['userSellingLocation'];

    //         // Check if the user's location is set to "all"
    //         if ($authUserSellingId == "all") {
    //             // Retrieve the latest debts for each customer
    //             $latestDebts = DB::table('supplier_debts')
    //                 ->select('supplier_id', DB::raw('MAX(created_at) as latest_entry'))
    //                 ->groupBy('supplier_id')
    //                 ->latest()
    //                 ->get();

    //             $supplierIds = $latestDebts->pluck('supplier_id')->toArray();

    //             $debts = SupplierDebts::whereIn('supplier_id', $supplierIds)
    //                 ->whereIn('created_at', $latestDebts->pluck('latest_entry')->toArray())
    //                 ->with('selling.location')->get();
    //         } else {
    //             // Retrieve the latest debts for each customer in the specified location
    //             $latestDebts = DB::table('supplier_debts')
    //                 ->select('supplier_id', DB::raw('MAX(created_at) as latest_entry'))
    //                 ->where('selling_id', $authUserSellingId)
    //                 ->groupBy('supplier_id')
    //                 ->latest()
    //                 ->get();

    //             $supplierIds = $latestDebts->pluck('supplier_id')->toArray();

    //             // Retrieve the specific debts based on the latest entry for each customer in the specified location
    //             $debts = SupplierDebts::whereIn('supplier_id', $supplierIds)
    //                 ->whereIn('created_at', $latestDebts->pluck('latest_entry')->toArray())
    //                 ->with('selling.location')->get();
    //         }

    //         $sum = 0;

    //         // Pass the retrieved data to the "debts-report" view
    //         return view("supplier-debts-report", compact("debts", "sum"));
    //     } catch (Exception $e) {
    //         // Log the error
    //         Log::error($e->getMessage());
    //         // Return a custom error response
    //         return response()->json(['error' => 'Something went wrong'], 500);
    //     }
    // }

    public function supplierDebts(Request $request)
    {
        try {
            # Get location details
            $locationDetails = processLocationDetails();
            $authUserSellingId = $locationDetails['userSellingLocation'];

            // Check if the user's location is set to "all"
            if ($authUserSellingId == "all") {
                // Retrieve the latest debts for each supplier
                $latestDebts = DB::table('supplier_debts')
                    ->select('supplier_id', DB::raw('MAX(created_at) as latest_entry'))
                    //->where('due', '>', 0)
                    ->groupBy('supplier_id')
                    ->latest()
                    ->get();

                $supplierIds = $latestDebts->pluck('supplier_id')->toArray();

                // Retrieve the specific debts based on the latest entry for each supplier
                $debts = SupplierDebts::whereIn('supplier_id', $supplierIds)
                    ->whereIn('created_at', $latestDebts->pluck('latest_entry')->toArray())
                    ->with('selling.location')
                    ->get();
            } else {
                // Retrieve the latest debts for each supplier in the specified location
                $latestDebts = DB::table('supplier_debts')
                    ->select('supplier_id', DB::raw('MAX(created_at) as latest_entry'))
                    //->where('due', '>', 0)
                    ->where('selling_id', $authUserSellingId)
                    ->groupBy('supplier_id')
                    ->latest()
                    ->get();

                $supplierIds = $latestDebts->pluck('supplier_id')->toArray();

                // Retrieve the specific debts based on the latest entry for each supplier in the specified location
                $debts = SupplierDebts::whereIn('supplier_id', $supplierIds)
                    ->whereIn('created_at', $latestDebts->pluck('latest_entry')->toArray())
                    ->with('selling.location')
                    ->get();
            }

            $sum = 0;

            // Pass the retrieved data to the "debts-report" view
            return view("supplier-debts-report", compact("debts", "sum"));
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function supplierStatement(Request $request, $supplierId, $sellingId)
    {
        try {
            # Get location details
            $locationDetails = processLocationDetails();
            $accounts = $locationDetails['accounts'];

            $latestSupplierDebts = SupplierDebts::where("supplier_id", $supplierId)
                ->where("selling_id", $sellingId)
                ->latest()
                ->first();

            $supplierDebtsQuery = supplierDebts::query();

            if ($request->duration === null) {
                $supplierDebtsQuery->where("supplier_id", $supplierId)
                    ->where("selling_id", $sellingId)
                    ->orderBy("created_at", "ASC")
                    ->get();
            } else {
                if ($request->duration === "today") {
                    $supplierDebtsQuery->where("supplier_id", $supplierId)
                        ->where("date", date('Y-m-d'))
                        ->where("selling_id", $sellingId)
                        ->orderBy("created_at", "ASC");
                } elseif ($request->duration === "set date") {
                    $supplierDebtsQuery->where("supplier_id", $supplierId)
                        ->where("date", $request->start)
                        ->where("selling_id", $sellingId)
                        ->orderBy("created_at", "ASC");
                } elseif ($request->duration === "set duration") {
                    $supplierDebtsQuery->where("supplier_id", $supplierId)
                        ->whereBetween("date", [$request->start, $request->end])
                        ->where("selling_id", $sellingId)
                        ->orderBy("created_at", "ASC");
                }
            }

            $profile = BusinessProfile::where('selling_id', $sellingId)->with('selling.location')->first();
            $debts = $supplierDebtsQuery->with('selling.location', 'supplier')->get();
            $sum = 0;

            return view("supplier-debts-history", compact(
                "sum",
                "debts",
                "profile",
                "accounts",
                "latestSupplierDebts"
            ));
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function purchasedItems($purchaseId)
    {
        try {
            $purchase = Purchase::find($purchaseId);
            $items = PurchasedItem::where("purchase_id", $purchaseId)->get();
            $profile = BusinessProfile::where('selling_id', $purchase->selling_id)->with('selling.location')->first();

            return view("purchased-items", compact("items", "purchase", "profile"));
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function sales(Request $request)
    {
        try {
            # Get location details
            $locationDetails = processLocationDetails();
            $sellingCount = $locationDetails['sellingCount'];
            $sellings = $locationDetails['sellings'];
            $accounts = $locationDetails['accounts'];

            $current_date = date('Y-m-d');
            $start = "";
            $end = "";
            $type = "";
            $status = "";
            $duration = "";
            $salesTotal = 0;
            $salesDiscount = 0;
            $salesNet = 0;
            $results = [];
            $discountSum = 0;
            $netSum = 0;
            $totalSum = 0;
            $proccessedItems = [];
            $tra = null;
            $overall_location_total = 0;
            $paid_amount = 0;
            $due_amount = 0;

            // Kitchen Varibales
            $checkWaiters = Waiter::count();
            $waiters = Waiter::all();
            $kitchenSales = 0;
            $counterSales = 0;
            $waiterLocationSale = 0;
            $waiter_total = 0;
            $overall_waiter_total = 0;

            // Return the sales-report view with the necessary data
            return view("sales-report", compact(
                "results",
                "start",
                "end",
                "duration",
                "sellings",
                "sellingCount",
                "checkWaiters",
                "current_date",
                "type",
                "salesTotal",
                "salesDiscount",
                "salesNet",
                "status",
                "proccessedItems",
                "accounts",
                "tra",
                "overall_location_total",
                "overall_waiter_total",
                "paid_amount",
                "due_amount",
                "waiters",
                "kitchenSales",
                "counterSales",
                "waiterLocationSale",
                "waiter_total"

            ));
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function vehicleReport(Request $request)
    {
        try {
            $loads = LoadVehicle::all();
            return view("vehicle-report", compact("loads"));
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function stockTacking(Request $request)
    {
        $stockTackingData = StockTacking::all();
        $missingValue = StockTacking::where('newqty', '<', 'oldqty')->sum('missing_value');
        return view("stock-tacking-report", compact(
            "stockTackingData",
            "missingValue"
        ));
    }

    public function getSalesReport(Request $request)
    {
        try {
            // Get current date
            $currentDate = date('Y-m-d');

            // Fetch data from the request with null coalescing operator
            $start = $request->start ?? null;
            $end = $request->end ?? null;
            $sellingId = $request->area ?? 'all';
            $type = $request->type ?? 'all';
            $duration = $request->duration ?? 'today';
            $status = $request->status ?? 'itemized';
            $waiterId = $request->waiter ?? null;

            // Validate dates if provided
            if ($start !== null && !strtotime($start)) {
                return response()->json(['error' => 'Invalid start date format.'], 400);
            }

            if ($end !== null && !strtotime($end)) {
                return response()->json(['error' => 'Invalid end date format.'], 400);
            }

            // Create a query object for retrieving results
            $query = Sale::with(
                "customer:id,name",
                "selling.location",
                "user:id,name",
                "account:id,name",
                "room:id,name",
                "waiter:id,name",
                "table:id,name"
            );

            // Apply filters based on duration
            if ($duration == "today") {
                $query->where("date", $currentDate);
            } elseif ($duration == "set date" && $start) {
                $query->where("date", $start);
            } elseif ($duration == "set duration" && $start && $end) {
                $query->whereBetween("date", [$start, $end]);
            }

            // Apply location filter if it is not "all"
            if ($sellingId != "all") {
                $data = $query->where("selling_id", $sellingId)->get();
            }

            // Apply waiter filter if it is not "null" & is not "all"
            if ($waiterId && $waiterId != "All") {
                $query->where("waiter_id", $waiterId);
            }

            // Apply type filter if it is not "all"
            if ($type != "all") {
                $query->where("status", $type);
            }

            // Calculate the sum of the "net" field directly in the query
            $salesTotal = (clone $query)->sum("total");
            $salesDiscount = (clone $query)->sum("discount");
            $salesNet = (clone $query)->sum("net");

            // Retrieve results with the order by clause
            $results = $query->orderBy("date", "asc")->get();

            // Get location details
            $locationDetails = processLocationDetails();
            $sellingCount = $locationDetails['sellingCount'];
            $sellings = $locationDetails['sellings'];
            $accounts = $locationDetails['accounts'];

            // Bar & Resturant Variable
            $checkWaiters = Waiter::count();
            $waiters = Waiter::all();
            $waiter = null;
            $waiter_total = 0;
            $overall_waiter_total = 0;
            $kitchenSales = 0;
            $counterSales = 0;

            $discountSum = 0;
            $netSum = 0;
            $totalSum = 0;
            $soldItems = [];
            $proccessedItems = [];

            $licenseData = License::select('tra')->first()->value('tra');
            $tra = null;

            if ($licenseData != null && auth()->user()->name == 'Admin') {
                $tra = $licenseData / 100;
            }

            // Pass the variables to the view using compact function
            return view("sales-report", compact(
                "results",
                "start",
                "end",
                "sellingId",
                "currentDate",
                "type",
                "duration",
                "sellings",
                "sellingCount",
                "accounts",
                "checkWaiters",
                "salesTotal",
                "salesDiscount",
                "salesNet",
                "status",
                "soldItems",
                "proccessedItems",
                "tra",
                "waiters",
                "waiter_total",
                "overall_waiter_total",
                "kitchenSales",
                "counterSales",
            ));
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function purchasesReportView()
    {
        try {
            $data = collect();

            # Get location details
            $locationDetails = processLocationDetails();
            $sellingCount = $locationDetails['sellingCount'];
            $sellings = $locationDetails['sellings'];

            return view("purchase-report", compact("data", "sellings", "sellingCount"));
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function getPurchasesReport(Request $request)
    {
        try {
            $start = $request->start ?? null;
            $end = $request->end ?? null;
            $sellingId = $request->area ?? 'all';
            $type = $request->type ?? 'today';

            $purchaseQuery = Purchase::with("supplier"); // Start building the query

            if ($sellingId != "all") {
                $purchaseQuery->where("selling_id", $sellingId); // Add location condition to the query if it's not "all"
            }

            if ($type == "today") {
                $purchaseQuery->whereDate("date", date('Y-m-d')); // Add condition to filter by today's date
            } elseif ($type == "set date") {
                $purchaseQuery->whereDate("date", $start); // Add condition to filter by the specified start date
            } elseif ($type == "set duration") {
                $purchaseQuery->whereBetween("date", [$start, $end]); // Add condition to filter by a date range
            }

            $data = $purchaseQuery->get(); // Execute the query and get the results

            # Get location details
            $locationDetails = processLocationDetails();
            $sellingCount = $locationDetails['sellingCount'];
            $sellings = $locationDetails['sellings'];

            return view("purchase-report", compact("data", "sellings", "sellingCount")); // Pass the data to the view and return the view
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function fetchSales(Request $request)
    {
        try {
            // Determine the status based on the request type
            $status = $request->type == "cash" ? "cash" : "credit";

            // Use the "when" method to conditionally apply different query scopes
            $data = Sale::when($request->type == "all", function ($query) {
                // If the type is "all", fetch all sales
                return $query->all();
            })->when($request->type != "all", function ($query) use ($status) {
                // If the type is "cash" or "credit", filter sales by the specified status
                return $query->where("status", $status);
            })->get();

            // Pass the fetched data to the "sales-report" view
            return view("sales-report", compact("data"));
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function fetchPurchases(Request $request, $id)
    {
        // Get the current date
        $currentDate = date('Y-m-d');

        // Fetch the purchase and eager load the related purchased items
        $purchase = Purchase::with('purchasedItems.unit', 'purchasedItems.product')->find($id);

        // Fetch the business profile
        $profile = BusinessProfile::where('selling_id', $purchase->selling_id)->first();

        // Fetch the supplier based on the purchase's supplier name
        $supplier = Supplier::find($purchase->supplier_id);

        # Get location details
        $locationDetails = processLocationDetails();
        $accounts = $locationDetails['accounts'];

        // Pass the necessary data to the invoice view
        return view('invoice', compact('currentDate', 'profile', 'purchase', 'supplier', 'accounts'));
    }

    public function fetchCustomerDebt($id)
    {
        try {
            // Fetch only the required columns from the BusinessProfile model using pluck
            $profile = BusinessProfile::select("address", "phone", "email", "name")->first();

            // Fetch the sale with the related customer using eager loading
            $sale = Sale::with('customers', 'items')->find($id);

            // Calculate the subtotal by summing the 'amount' column of the items
            $subtotal = $sale->items->sum('amount');

            // Get the current date
            $currentDate = date('Y-m-d');

            // Pass the necessary variables to the view
            return view("new-invoice", [
                'id' => $id, // The ID of the sale
                'sale' => $sale, // The sale model with the related customer
                'profile' => $profile, // The business profile
                'currentDate' => $currentDate, // The current date
                'customer' => $sale->customer, // The customer related to the sale
                'subtotal' => $subtotal, // The calculated subtotal
            ]);
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function printSaleInvoice($id)
    {
        try {
            // Get the current date
            $currentDate = date('Y-m-d');

            // Fetch the sale and eager load the related sold items
            $sale = Sale::with('items', 'customers')->find($id);

            // Fetch the business profile
            $profile = BusinessProfile::select('address', 'phone', 'email', 'name')->first();

            // Calculate the subtotal by summing the sold item amounts

            $subtotal = $sale->items->sum('amount');

            // Pass the necessary data to the new-invoice-print view
            return view('new-invoice-print', compact('id', 'sale', 'profile', 'currentDate', 'subtotal'));
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function printPurchaseInvoice($id)
    {
        try {
            // Get the current date
            $currentDate = date('Y-m-d');

            // Fetch the purchase and eager load the related purchased items
            $purchase = Purchase::with('items')->find($id);

            // Fetch the business profile
            $profile = BusinessProfile::select('address', 'phone', 'email', 'name')->first();

            // Fetch the supplier based on the purchase's supplier name
            $supplier = Supplier::where('name', $purchase->supplier)->first();

            // Calculate the subtotal by summing the purchased item amounts
            $subtotal = $purchase->purchasedItems->sum('amount');

            // Pass the necessary data to the new-purchase-invoice-print view
            return view('new-purchase-invoice-print', compact('id', 'purchase', 'profile', 'currentDate', 'supplier', 'subtotal'));
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function fetchWaiterDebt(Request $request, $id)
    {
        try {
            return view("waiter-debt", compact("id"));
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function StockBalance(Request $request)
    {
        try {
            # Get location details
            $locationDetails = processLocationDetails();
            $sellingCount = $locationDetails['sellingCount'];
            $sellings = $locationDetails['sellings'];

            $locationId = $request->location;
            $products = Product::where("location_id", $locationId)
                ->with("unit", "stock", "category")->get();
            $area = Location::find($locationId);

            return view("stock-balance-report", compact("products", "area", "sellings", "sellingCount", "locationId"));
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }


    public function stockValue(Request $request)
    {
        try {
            # Get location details
            $locationDetails = processLocationDetails();
            $sellingCount = $locationDetails['sellingCount'];
            $sellings = $locationDetails['sellings'];

            // Retrieve the location ID from the request input.
            $locationId = $request->input('location');

            // Check if a valid location ID is provided in the request.
            if (!$locationId) {
                // Redirect back with an error message if no valid location ID is selected.
                return redirect()->back()->with('error', 'Please select a valid location.');
            }

            // Find the "Menu" category in the ProductCategory table.
            $category = ProductCategory::where("name", "Menu")->first();

            // Create a query to retrieve Product records with their stock data.
            $productsQuery = Product::where("location_id", $locationId)
                ->whereHas('stock', function ($query) {
                    $query->where('qty', '>=', 0);
                })
                ->withSum('stock', 'qty')
                ->orderBy('id', 'asc');

            // If the "Menu" category exists, exclude it from the query results.
            if ($category) {
                $productsQuery->where("category_id", "!=", $category->id);
            }

            // Get the products based on the location and category filtering.
            $products = $productsQuery->get();

            // Find the location based on the provided location ID.
            $area = Location::find($locationId);

            $licenseData = License::select('tra')->first()->value('tra');
            $tra = null;

            if ($licenseData != null && auth()->user()->name == 'Admin') {
                $tra = $licenseData / 100;
            }

            // Render the view with the retrieved data.
            return view("stock-value-report", compact("products", "locationId", "area", "sellings", "sellingCount", "tra"));
        } catch (\Exception $e) {
            // Redirect back with an error message if an exception occurs.
            return redirect()->back()->with('error', 'An error occurred: ' . $e->getMessage());
        }
    }

    public function openingClosing(Request $request)
    {
        try {
            $type = $request->type;
            $currentDate = date('Y-m-d');
            $start = $request->start;
            $end = $request->end;
            $area = $request->area;
            $stocks = Stock::where("location", $area)
                ->orderBy("product", "asc")
                ->get();

            return view("opening-and-closing", compact("area", "start", "end", "type", "currentDate", "stocks"));
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function stockMovement(Request $request)
    {
        # Get location details
        $locationDetails = processLocationDetails();
        $sellingCount = $locationDetails['sellingCount'];
        $sellings = $locationDetails['sellings'];

        // Get the current date
        $currentDate = date('Y-m-d');

        // Get the request parameters
        $type       = $request->type;
        $startDate  = $request->start;
        $endDate    = $request->end;
        $locationId = $request->area;

        # Handle duration logic
        if ($type === 'today') {
            $startDate = $currentDate;
            $endDate   = $currentDate;
        } elseif ($type === 'set date') {
            // Use startDate only
            $endDate = $startDate;
        } elseif ($type === 'set duration') {
            // Use both startDate and endDate as provided (no change needed)
        } else {
            // Default fallback: use current date
            $startDate = $currentDate;
            $endDate   = $currentDate;
        }

        // Get the required category ids
        $categoryIds = ProductCategory::where('parent_id', 3)
            ->pluck('id')
            ->toArray();

        // Fetch all products, apply category filter only if $categoryIds is not empty
        $products = Product::with('stock')
            ->when(!empty($categoryIds), function ($query) use ($categoryIds) {
                $query->whereIn('category_id', $categoryIds);
            })
            ->get();

        $report = $products->map(function ($product) use ($startDate, $endDate, $locationId) {
            $query = StockMovement::where('product_id', $product->id)
                ->when($locationId, fn($q) => $q->where('location_id', $locationId));

            // Opening stock: before start date
            $opening = (clone $query)->where('date', '<', $startDate)->sum('qty');

            // Current stock movements in the range
            $current = StockMovement::where('product_id', $product->id)
                ->whereBetween('date', [$startDate, $endDate])
                ->when($locationId, fn($q) => $q->where('location_id', $locationId));

            $purchases   = (clone $current)->where('reason', 'purchase')->sum('qty');
            $sales       = (clone $current)->where('reason', 'sale')->sum('qty');
            $damages     = (clone $current)->where('reason', 'damage')->sum('qty');
            $transfers   = (clone $current)->where('reason', 'transfer')->sum('qty');
            $adjustments = (clone $current)->where('reason', 'adjustment')->sum('qty');

            // Closing stock calculation
            $closing = $opening + $purchases - $sales - $damages + $transfers + $adjustments;

            return [
                'product'     => $product->product,
                'opening'     => $opening,
                'purchases'   => $purchases,
                'sales'       => $sales,
                'damages'     => $damages,
                'transfers'   => $transfers,
                'adjustments' => $adjustments,
                'closing'     => $closing,
            ];
        });

        // Pass the data to the Blade view
        return view('stock-movement-report', compact(
            'report',
            'startDate',
            'endDate',
            'sellings',
            'sellingCount'
        ));
    }

    public function fastSlowProudcts(Request $request)
    {
        # Get location details
        $locationDetails = processLocationDetails();
        $sellingCount = $locationDetails['sellingCount'];
        $sellings = $locationDetails['sellings'];

        # Get the selected location id.
        $sellingId = $request->location;

        $type = $request->type;
        $currentDate = date('Y-m-d');
        $start = $request->start;
        $end = $request->end;

        # Get the top moving Items 
        $topMovingProductsQuery = DB::table('sold_items')
            ->join('sales', 'sold_items.sale_id', '=', 'sales.id')
            ->join('products', 'sold_items.product_id', '=', 'products.id')
            ->join('units', 'sold_items.unit_id', '=', 'units.id')
            ->select(
                'products.product as name',
                'units.name as unit',
                DB::raw('SUM(sold_items.qty) as total_qty'),
                DB::raw('SUM(sold_items.amount) as total_amount'),
                DB::raw('MAX(sold_items.price) as price')
            );

        // Apply selling_id filter if needed
        if ($sellingId !== 'all') {
            $topMovingProductsQuery->where('sales.selling_id', $sellingId);
        }

        // Optional: add date filters
        if ($type === 'today') {
            $topMovingProductsQuery->where('sales.date', $currentDate);
        } elseif ($type === 'set date') {
            $topMovingProductsQuery->where('sales.date', $start);
        } else {
            $topMovingProductsQuery->whereBetween('sales.date', [$start, $end]);
        }

        $products = $topMovingProductsQuery
            ->groupBy('sold_items.product_id', 'products.product', 'units.name')
            ->orderByDesc('total_qty')
            ->get();


        // Pass the data to a Blade template for rendering
        return view('fast-and-slow-moving-products', compact('products', 'sellings'));
    }


    public function stockReport(Request $request)
    {
        try {
            # Get location details
            $locationDetails = processLocationDetails();
            $sellingCount = $locationDetails['sellingCount'];
            $sellings = $locationDetails['sellings'];

            $area = "";
            $products = collect();
            $results = collect();
            $report = collect();
            $type = $request->type;
            $tra = null;

            if ($type == "damage") {
                return view("damage-stock-report", compact(
                    "results",
                    "sellings",
                    "sellingCount"
                ));
            } elseif ($type == "transfer") {
                return view("stock-transfer-report", compact(
                    "results",
                    "sellings",
                    "sellingCount"
                ));
            } elseif ($type == "value") {
                return view("stock-value-report", compact(
                    "products",
                    "area",
                    "sellings",
                    "sellingCount",
                    "tra"
                ));
            } elseif ($type == "movement") {
                return view("stock-movement-report", compact(
                    "report",
                    "sellings",
                    "sellingCount"
                ));
            } elseif ($type == "balance") {
                return view("stock-balance-report", compact(
                    "area",
                    "products",
                    "sellings",
                    "sellingCount"
                ));
            } elseif ($type == "fast & slow") {
                return view("fast-and-slow-moving-products", compact(
                    "area",
                    "products",
                    "sellings",
                    "sellingCount"
                ));
            }

            return view("stock-report");
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function incomeReport()
    {
        try {
            // -------------------------
            // 1. Location details
            // -------------------------
            $locationDetails = processLocationDetails();
            $sellingCount = $locationDetails['sellingCount'];
            $sellings = $locationDetails['sellings'];

            // -------------------------
            // 2. Default values
            // -------------------------
            $currentDate = date('Y-m-d');
            $start = "";
            $end = "";
            $cashSales = 0;
            $creditSales = 0;
            $totalSales = 0;
            $discount = 0;
            $salesReturn = 0;
            $salesDeductions = 0;
            $netSales = 0;
            $cogs = 0;
            $grossProfit = 0;
            $operatingExpenses = 0;
            $administrativeExpenses = 0;
            $sellingExpenses = 0;
            $financeExpenses = 0;
            $extraordinaryExpenses = 0;
            $operatingProfit = 0;
            $vat = 0;
            $netProfit = 0;
            $tra = null;

            // -------------------------
            // 3. Render view with zeros
            // -------------------------
            return view("income-statement", compact(
                'cashSales',
                'creditSales',
                'totalSales',
                'discount',
                'salesReturn',
                'salesDeductions',
                'netSales',
                'cogs',
                'grossProfit',
                'operatingExpenses',
                'administrativeExpenses',
                'sellingExpenses',
                'financeExpenses',
                'extraordinaryExpenses',
                'operatingProfit',
                'vat',
                'netProfit',
                'currentDate',
                'start',
                'end',
                'sellings',
                'sellingCount',
                'tra'
            ));
        } catch (Exception $e) {
            Log::error($e->getMessage());
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }


    public function getIncomeReport(Request $request)
    {
        try {
            // -------------------------
            // 1. Location details
            // -------------------------
            $locationDetails = processLocationDetails();
            $sellingCount = $locationDetails['sellingCount'];
            $sellings = $locationDetails['sellings'];

            $currentDate = date('Y-m-d');
            $start = $request->start;
            $end = $request->end;
            $sellingId = $request->area;
            $type = $request->type;

            // -------------------------
            // 2. Sold items / COGS
            // -------------------------
            $soldItemsQuery = DB::table('sales')
                ->join('sold_items', 'sales.id', '=', 'sold_items.sale_id');

            if ($sellingId !== 'all') {
                $soldItemsQuery->where('sales.selling_id', $sellingId);
            }

            if ($type === 'today') {
                $soldItemsQuery->where('sales.date', $currentDate);
            } elseif ($type === 'set date') {
                $soldItemsQuery->where('sales.date', $start);
            } else {
                $soldItemsQuery->whereBetween('sales.date', [$start, $end]);
            }

            $soldItems = $soldItemsQuery->select('sold_items.product_id', 'sold_items.qty', 'sold_items.cont')->get();
            $productIds = $soldItems->pluck('product_id')->unique();
            $buyingPrices = Product::whereIn('id', $productIds)->pluck('buying', 'id');

            $cogs = $soldItems->sum(function ($item) use ($buyingPrices) {
                return ($buyingPrices[$item->product_id] ?? 0) * $item->qty * $item->cont;
            });

            // -------------------------
            // 3. Sales / Returns / Discounts
            // -------------------------
            $saleQuery = Sale::query();
            $salesReturnQuery = SaleReturn::query();
            $expensesQuery = PostExpense::with('category');

            if ($sellingId !== 'all') {
                $saleQuery->where('selling_id', $sellingId);
                $salesReturnQuery->where('selling_id', $sellingId);
                $expensesQuery->where('selling_id', $sellingId);
            }

            if ($type === 'today') {
                $saleQuery->where('date', $currentDate);
                $salesReturnQuery->where('date', $currentDate);
                $expensesQuery->where('date', $currentDate);
            } elseif ($type === 'set date') {
                $saleQuery->where('date', $start);
                $salesReturnQuery->where('date', $start);
                $expensesQuery->where('date', $start);
            } else {
                $saleQuery->whereBetween('date', [$start, $end]);
                $salesReturnQuery->whereBetween('date', [$start, $end]);
                $expensesQuery->whereBetween('date', [$start, $end]);
            }

            $cashSales = (clone $saleQuery)->where('status', 'cash')->sum('total');
            $creditSales = (clone $saleQuery)->where('status', 'credit')->sum('total');
            $discount = (clone $saleQuery)->sum('discount');
            $salesReturn = $salesReturnQuery->sum('total');

            $totalSales = $cashSales + $creditSales;
            $salesDeductions = $discount + $salesReturn;
            $netSales = $totalSales - $salesDeductions;
            $grossProfit = $netSales - $cogs;

            // -------------------------
            // 4. Expense type aggregation
            // -------------------------
            $expenses = $expensesQuery->get();

            // Initialize expense type totals
            $operatingExpenses = 0;
            $administrativeExpenses = 0;
            $sellingExpenses = 0;
            $financeExpenses = 0;
            $extraordinaryExpenses = 0;

            foreach ($expenses as $expense) {
                $type = $expense->category->expense_type ?? 'operating';
                switch ($type) {
                    case 'operating':
                        $operatingExpenses += $expense->sum;
                        break;
                    case 'administrative':
                        $administrativeExpenses += $expense->sum;
                        break;
                    case 'selling':
                        $sellingExpenses += $expense->sum;
                        break;
                    case 'finance':
                        $financeExpenses += $expense->sum;
                        break;
                    case 'extraordinary':
                        $extraordinaryExpenses += $expense->sum;
                        break;
                }
            }

            // Determine what to show based on duration
            if ($request->type === 'today') {
                $administrativeExpenses = 0;
                $sellingExpenses = 0;
                $financeExpenses = 0;
                $extraordinaryExpenses = 0;
            } elseif ($request->type === 'set date') {
                $financeExpenses = 0;
                $extraordinaryExpenses = 0;
            }
            // Monthly / longer durations: show all

            $operatingProfit = $grossProfit - $operatingExpenses - $administrativeExpenses - $sellingExpenses;

            $vat = (clone $saleQuery)->sum('vat');
            $netProfit = $operatingProfit - $financeExpenses - $extraordinaryExpenses - $vat;

            // TRA logic
            $licenseData = License::value('tra');
            $tra = (auth()->user()->name === 'Admin' && $licenseData) ? ($licenseData / 100) : null;

            return view('income-statement', compact(
                'cashSales',
                'creditSales',
                'totalSales',
                'discount',
                'salesReturn',
                'salesDeductions',
                'netSales',
                'cogs',
                'grossProfit',
                'operatingExpenses',
                'administrativeExpenses',
                'sellingExpenses',
                'financeExpenses',
                'extraordinaryExpenses',
                'operatingProfit',
                'vat',
                'netProfit',
                'currentDate',
                'start',
                'end',
                'sellings',
                'sellingCount',
                'tra'
            ));
        } catch (Exception $e) {
            Log::error($e->getMessage());
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function cashMovementView()
    {
        try {
            # Get location details
            $locationDetails = processLocationDetails();
            $sellingCount = $locationDetails['sellingCount'];
            $sellings = $locationDetails['sellings'];

            $type = "";
            $start = "";
            $end = "";
            $results = [];

            return view("cash-movement-report", compact(
                "type",
                "start",
                "end",
                "results",
                "sellings",
                "sellingCount"
            ));
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function cashMovement(Request $request)
    {
        try {
            $currentDate = date('Y-m-d'); // Get the current date
            $start = $request->start; // Get the start date from the request
            $end = $request->end; // Get the end date from the request
            $type = $request->type; // Get the type from the request

            # Get location details
            $locationDetails = processLocationDetails();
            $sellingCount = $locationDetails['sellingCount'];
            $sellings = $locationDetails['sellings'];

            $query = Transaction::query();

            // Filter by selling id if not "all"
            if (auth()->user()->selling_id !== 'all') {
                $query->where('selling_id', auth()->user()->selling_id);
            }

            // Create the base query for transactions
            $query = Transaction::with("user", "account", "selling.location")
                ->orderBy('created_at', 'ASC');

            if ($type === 'today') {
                // Fetch transactions for the current date
                $query->whereDate('date', date('Y-m-d'));
            } elseif ($type === 'set date') {
                // Fetch transactions for the specified start date
                $query->whereDate('date', $start);
            } elseif ($type === 'set duration') {
                // Fetch transactions within the specified date range
                $query->whereDate('date', '>=', $start)
                    ->whereDate('date', '<=', $end);
            }

            // Retrieve the results of the query
            $results = $query->get();

            // Pass the necessary variables to the view
            return view('cash-movement-report', compact(
                'start',
                'end',
                'type',
                'results',
                'sellings',
            ));
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        //
    }

    public function customersReport()
    {
        try {
            $customers = Booking::all();
            return view('customers-report', compact('customers'));
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function openingClosingStock()
    {
        return view("opening-and-closing");
    }

    public function salesReportPendingDebts($sellingId, $duration, $start, $end)
    {
        // Prepare query
        $salesQuery = Sale::selectRaw('customer_id, sum(net) as total_debt, sum(paid) as total_paid, sum(due) as total_due')
            ->where('status', 'credit')
            ->groupBy('customer_id');

        // Apply filters based on duration
        if ($duration == 'today') {
            // If duration is "today", fetch sales with date equal to current date
            $salesQuery->where('date', date('Y-m-d'));
            $message = "Today's pending debts.";
        } elseif ($duration == 'set date') {
            // If duration is "set date", fetch sales with date equal to the start date
            $message = $start . " pending debts.";
            $salesQuery->where('date', $start);
        } elseif ($duration == 'set duration') {
            // If duration is "set duration", fetch sales between start and end date
            $message = "Pending debts between " . $start . " - " . $end;
            $salesQuery->whereBetween('date', [$start, $end]);
        }

        // Get results
        $sales = $salesQuery->get();

        $total_debt = 0;
        $total_paid = 0;
        $total_due = 0;

        return view('sales-report-pending-debts', compact(
            'sales',
            'total_debt',
            'total_paid',
            'total_due',
            'message',
        ));
    }

    public function generatePdf($personId, $sellingId, $duration, $startDate, $endDate, $reportType, $saleId, $entityType)
    {
        // Determine if we're dealing with a customer or a supplier
        $isCustomer = ($entityType === "customer");

        // Get the person (Customer or Supplier)
        $person = $isCustomer
            ? Customer::find($personId)
            : Supplier::find($personId);

        if (!$person) {
            return redirect()->back()->with('error', ucfirst($entityType) . ' not found.');
        }

        // Get the business profile
        $businessProfile = BusinessProfile::where('selling_id', $person->selling_id)->first();

        if ($reportType == 1) {
            // Fetch debts depending on entity type
            $debts = $isCustomer
                ? Debts::where('customer_id', $personId)
                ->where('selling_id', $sellingId)
                : SupplierDebts::where('supplier_id', $personId)
                ->where('selling_id', $sellingId);

            // Apply date filter
            $debts = $debts->when($duration === 'today', fn($query) => $query->whereDate('date', now()))
                ->when($duration === 'set date', fn($query) => $query->whereDate('date', $startDate))
                ->when($duration === 'set duration', fn($query) => $query->whereBetween('date', [$startDate, $endDate]))
                ->get();

            // Get transaction IDs
            $transactionIds = $isCustomer
                ? $debts->pluck('sale_id')->toArray()
                : $debts->pluck('purchase_id')->toArray();

            // Load transactions
            $transactionModel = $isCustomer
                ? Sale::whereIn('id', $transactionIds)
                : Purchase::whereIn('id', $transactionIds);

            // Totals
            $totalAmount = $transactionModel->sum('total');
            $totalDiscount = $transactionModel->sum('discount');
            $totalPaid = $transactionModel->sum('paid');
            $totalDue = $totalAmount - $totalDiscount - $totalPaid;

            // Get transaction details
            if ($isCustomer) {
                $transactions = DB::table('sales')
                    ->whereIn('sales.id', $transactionIds)
                    ->join('sold_items', 'sales.id', '=', 'sold_items.sale_id')
                    ->join('products', 'sold_items.product_id', '=', 'products.id')
                    ->join('units', 'sold_items.unit_id', '=', 'units.id')
                    ->select('sales.*', 'sold_items.*', 'products.product as product_name', 'units.name as unit_name')
                    ->get();
            } else {
                $transactions = DB::table('purchases')
                    ->whereIn('purchases.id', $transactionIds)
                    ->join('purchased_items', 'purchases.id', '=', 'purchased_items.purchase_id')
                    ->join('products', 'purchased_items.product_id', '=', 'products.id')
                    ->join('units', 'purchased_items.unit_id', '=', 'units.id')
                    ->select('purchases.*', 'purchased_items.*', 'products.product as product_name', 'units.name as unit_name')
                    ->get();
            }

            return view('generate-pdf', [
                'entityType'    => $entityType,
                'person'        => $person,
                'transactions'  => $transactions,
                'businessProfile' => $businessProfile,
                'totalAmount'   => $totalAmount,
                'totalDiscount' => $totalDiscount,
                'totalPaid'     => $totalPaid,
                'totalDue'      => $totalDue,
            ]);
        } elseif ($reportType == 2) {
            // Get itemized single sale or purchase
            if ($isCustomer) {
                $transaction = Sale::with(['soldItems.product', 'soldItems.unit', 'customer', 'selling'])
                    ->find($saleId);
            } else {
                $transaction = Purchase::with(['purchasedItems.product', 'purchasedItems.unit', 'supplier', 'selling'])
                    ->find($saleId);
            }

            if (!$transaction) {
                abort(404, ucfirst($entityType) . ' transaction not found.');
            }

            $items = $isCustomer ? $transaction->soldItems : $transaction->purchasedItems;

            return view('generate-itemized-pdf', [
                'entityType'      => $entityType,
                'transaction'     => $transaction,
                'businessProfile' => $businessProfile,
                'person'          => $person,
                'items'           => $items,
            ]);
        }

        return redirect()->back()->with('error', 'Invalid report type');
    }

    public function accountingOverview(Request $request)
    {
        $type = $request->input('type', 'today'); // today, this_week, this_month, custom
        $start = $request->input('start');
        $end = $request->input('end');

        // Set date range
        $fromDate = now()->startOfDay();
        $toDate = now()->endOfDay();

        if ($type == 'this_week') {
            $fromDate = now()->startOfWeek();
            $toDate = now()->endOfWeek();
        } elseif ($type == 'this_month') {
            $fromDate = now()->startOfMonth();
            $toDate = now()->endOfMonth();
        } elseif ($type == 'custom' && $start) {
            $fromDate = Carbon::parse($start)->startOfDay();
            $toDate = $end ? Carbon::parse($end)->endOfDay() : $fromDate->copy()->endOfDay();
        }

        $dateLabel = $fromDate->format('Y-m-d');
        if ($fromDate != $toDate) {
            $dateLabel .= ' to ' . $toDate->format('Y-m-d');
        }

        // =======================
        // AGGREGATE DATA QUERIES
        // =======================

        // Cash / Bank
        $cash = DB::table('transactions')
            ->whereBetween('date', [$fromDate, $toDate])
            ->sum('after');

        // Customer Debts
        $customerDebts = DB::table('debts')
            ->whereBetween('date', [$fromDate, $toDate])
            ->sum('due');

        // Inventory
        $inventoryValue = DB::table('stocks')
            ->join('products', 'stocks.product_id', '=', 'products.id')
            ->select(DB::raw('SUM(stocks.qty * products.buying) as total'))
            ->value('total') ?? 0;

        // Fixed Assets
        $fixedAssets = Asset::sum('value');

        $totalAssets = $cash + $customerDebts + $inventoryValue + $fixedAssets;

        // Supplier Debts
        $supplierDebts = DB::table('supplier_debts')
            ->whereBetween('date', [$fromDate, $toDate])
            ->sum('due');

        $totalLiabilities = $supplierDebts;

        // =======================
        // PROFIT & LOSS
        // =======================

        $totalSales = Sale::whereBetween('date', [$fromDate, $toDate])->sum('net');
        $totalPurchases = Purchase::whereBetween('date', [$fromDate, $toDate])->sum('net');
        $totalExpenses = PostExpense::whereBetween('date', [$fromDate, $toDate])->sum('sum');

        // Net Profit = Sales - (Purchases + Expenses)
        $netProfit = $totalSales - ($totalPurchases + $totalExpenses);
        $totalEquity = $netProfit;

        // =======================
        // BALANCE CHECK
        // =======================
        $balance = $totalAssets - ($totalLiabilities + $totalEquity);

        // =======================
        // TRIAL BALANCE
        // =======================
        $trialBalance = [
            ['account' => 'Cash / Bank', 'debit' => $cash, 'credit' => 0],
            ['account' => 'Customer Debts (A/R)', 'debit' => $customerDebts, 'credit' => 0],
            ['account' => 'Inventory', 'debit' => $inventoryValue, 'credit' => 0],
            ['account' => 'Fixed Assets', 'debit' => $fixedAssets, 'credit' => 0],
            ['account' => 'Supplier Debts (A/P)', 'debit' => 0, 'credit' => $supplierDebts],
            ['account' => 'Retained Earnings', 'debit' => 0, 'credit' => $netProfit],
        ];

        $totalDebit = collect($trialBalance)->sum('debit');
        $totalCredit = collect($trialBalance)->sum('credit');

        // =======================
        // RETURN VIEW
        // =======================
        return view('accounting-overview', compact(
            'dateLabel',
            'cash',
            'customerDebts',
            'inventoryValue',
            'fixedAssets',
            'totalAssets',
            'supplierDebts',
            'totalLiabilities',
            'netProfit',
            'totalEquity',
            'totalSales',       // add this
            'totalPurchases',   // add this
            'totalExpenses',    // add this
            'balance',
            'trialBalance',
            'totalDebit',
            'totalCredit',
            'type',
            'start',
            'end'
        ));
    }

    public function settlementReport(Request $request)
    {
        //dd($request->all());
        $sellingId = $request->area;
        $duration = $request->type;

        // Initialize Variables
        $startDate = '';
        $currentDate = date('Y-m-d');
        $endDate = '';
        $totalSale = 0;
        $totalCashSale = 0;
        $totalCreditSale = 0;
        $totalExpenses = 0;
        $totalCollection = 0;
        $pendingDebts = [];
        $paidDebts = [];
        $totalPendingDebt = 0;
        $totalPaidDebt = 0;

        // Set start and end dates based on the duration
        if ($duration == "today") {
            $startDate = $currentDate;
        } elseif ($duration == "set date") {
            $startDate = $request->start;
        } elseif ($duration == "set duration") {
            $startDate = $request->start;
            $endDate = $request->end;
        }

        // Define common query for sales and expenses
        $saleQuery = Sale::where('selling_id', $sellingId);
        $expenseQuery = PostExpense::where('selling_id', $sellingId);
        $debtQuery = Debts::where('selling_id', $sellingId);

        // Apply date filters
        if ($duration == "today") {
            $saleQuery->where('date', $currentDate);
            $expenseQuery->where('date', $currentDate);
            $debtQuery->where('date', $currentDate);
        } elseif ($duration == "set date") {
            $saleQuery->where('date', $startDate);
            $expenseQuery->where('date', $startDate);
            $debtQuery->where('date', $startDate);
        } elseif ($duration == "set duration") {
            $saleQuery->whereBetween('date', [$startDate, $endDate]);
            $expenseQuery->whereBetween('date', [$startDate, $endDate]);
            $debtQuery->whereBetween('date', [$startDate, $endDate]);
        }

        // Calculate totals
        $totalSale = $saleQuery->sum('net') ?? 0;
        $totalCashSale = (clone $saleQuery)->where('status', 'cash')->sum('net') ?? 0;
        $totalCreditSale = (clone $saleQuery)->where('status', 'credit')->sum('net') ?? 0;
        $totalExpenses = $expenseQuery->sum('sum') ?? 0;
        $totalCollection =  (clone $debtQuery)->where('paid', '>', 0)->sum('paid') ?? 0;
        $customerIds = (clone $debtQuery)->pluck('customer_id')->unique(); # Get customers who paid.
        $salesCollection = (clone $saleQuery)->whereIn('customer_id', $customerIds);

        // Get Pending Debts
        $pendingDebts = $saleQuery->with(['customer' => function ($query) {
            $query->withTrashed(); // Include soft-deleted customers
        }])
            ->where('status', 'credit')
            ->get();

        // Get Paid Debts
        $paidDebts = $debtQuery->with([
            'customer' => function ($query) {
                $query->withTrashed(); // Include soft-deleted customers
            },
            'account' => function ($query) {
                $query->withTrashed(); // Include soft-deleted accounts
            }
        ])
            ->where('paid', '>', 0)
            ->get();

        // Calculate total debts
        $totalPendingDebt = $pendingDebts->sum('pending_debt');
        $totalPaidDebt = $paidDebts->sum('paid_debt');

        // Calculate Final Balance
        $finalBalance = ($totalSale - ($totalExpenses + $totalCollection));

        # Get location details
        $locationDetails = processLocationDetails();
        $sellings = $locationDetails['sellings'];

        return view('settlement-report', compact(
            'totalSale',
            'totalCashSale',
            'totalCreditSale',
            'totalExpenses',
            'totalCollection',
            'finalBalance',
            'sellings',
            'totalPendingDebt',
            'totalPaidDebt',
            'pendingDebts',
            'paidDebts',
        ));
    }

    public function inventoryReport(Request $request)
    {
        $locationId = 1; // Example; can use Auth::user()->location_id

        // Date range from request
        $from = '2025-01-01';
        $to = '2025-01-31';

        $products = Product::with('unit', 'stock')
            ->where('location_id', $locationId)
            ->get();

        $reportData = $products->map(function ($product) use ($from, $to, $locationId) {
            $opening = $product->stock->qty ?? 0;

            // Purchases within duration
            $purchases = $product->purchasedItems()
                ->whereHas('purchase', function ($q) use ($from, $to, $locationId) {
                    $q->whereBetween('date', [$from, $to])
                        ->where('storage_id', $locationId);
                })->sum('qty');

            // Purchase Returns within duration
            $purchaseReturns = $product->purchaseReturnItems()
                ->whereHas('purchaseReturn', function ($q) use ($from, $to, $locationId) {
                    $q->whereBetween('date', [$from, $to])
                        ->where('selling_id', $locationId);
                })->sum('qty');

            // Sales within duration
            $sales = $product->soldItems()
                ->whereHas('sale', function ($q) use ($from, $to, $locationId) {
                    $q->whereBetween('date', [$from, $to])
                        ->where('selling_id', $locationId);
                })->sum('qty');

            // Sales Returns within duration
            $salesReturns = $product->salesReturnItems()
                ->whereHas('salesReturn', function ($q) use ($from, $to, $locationId) {
                    $q->whereBetween('date', [$from, $to])
                        ->where('selling_id', $locationId);
                })->sum('qty');

            // Damages within duration
            $damages = $product->damages()
                ->where('location_id', $locationId)
                ->whereBetween('date', [$from, $to])
                ->sum('qty');

            // Calculate closing and sold
            $closing = $opening + $purchases - $purchaseReturns - $sales + $salesReturns - $damages;
            $sold = ($opening + $purchases - $purchaseReturns - $damages) - $closing - $salesReturns;

            return [
                'product' => $product->product,
                'unit' => $product->unit->name ?? '',
                'opening' => $opening,
                'purchase' => $purchases,
                'purchase_return' => $purchaseReturns,
                'damage' => $damages,
                'sold' => $sold,
                'sales_return' => $salesReturns,
                'adjustment' => 0,
                'closing' => $closing,
            ];
        });

        return view('inventory.report', compact('reportData', 'from', 'to'));
    }

    // Helper function to calculate the previous day's final balance dynamically
    // private function calculatePreviousFinalBalance($date, $location)
    // {
    //     // Calculate previous day's cash sales
    //     $previousDayCashSales = Sale::where('location_id', $location)
    //         ->where('date', $date)
    //         ->where('status', 'cash')
    //         ->sum('net');

    //     // Calculate previous day's credit sales
    //     $previousDayCreditSales = Sale::where('location_id', $location)
    //         ->where('date', $date)
    //         ->where('status', 'credit')
    //         ->sum('net');

    //     // Calculate previous day's expenses
    //     $previousDayExpenses = PostExpense::where('location_id', $location)
    //         ->where('date', $date)
    //         ->sum('sum');

    //     // Calculate previous day's collections (paid debts)
    //     $previousDayCollections = Debts::where('location_id', $location)
    //         ->where('date', $date)
    //         ->sum('paid');

    //     // Final balance is cash sales + credit sales - expenses + collections
    //     $previousFinalBalance = ($previousDayCashSales + $previousDayCreditSales - $previousDayExpenses + $previousDayCollections);

    //     return $previousFinalBalance;
    // }
}
