<?php

namespace App\Http\Controllers;

use App\Imports\HotelmaxSalesImport;
use App\Imports\SalesImport;
use App\Imports\SoldItemsImport;
use App\Models\Sale;
use App\Models\Stock;
use App\Models\SoldItem;
use App\Models\Transaction;
use App\Models\Account;
use App\Models\BusinessProfile;
use App\Models\Customer;
use App\Models\Debts;
use App\Models\LoadedItems;
use App\Models\LoadVehicle;
use App\Models\Product;
use App\Models\ProductCategory;
use App\Models\Profoma;
use App\Models\SaleReturn;
use App\Models\SalesReturnItem;
use App\Models\Selling;
use App\Models\ServiceProduct;
use App\Models\StockMovement;
use App\Models\Table;
use App\Models\UnitAssigned;
use App\Models\Waiter;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Maatwebsite\Excel\Facades\Excel;
use Illuminate\Support\Facades\Session;
use function updateCustomerDebts; // Import the function

class SaleController extends Controller
{

    public function __construct()
    {
        $this->middleware('auth');
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */

    public function index($id, $status)
    {
        try {
            if ($status == 'profoma') {
                $profomaId = $id;

                // Retrieve profoma data and items from session if available
                $profomaData = Profoma::with('profomaItems.unit', 'profomaItems.product')->find($profomaId);
                $profomaItems = $profomaData->profomaItems;

                // Add the two variables to their respective sessions
                session(['profomaData' => $profomaData]);
                session(['profomaItems' => $profomaItems]);
            } else {
                session()->forget(['profomaData', 'profomaItems', 'customerData']);
            }

            # Get location details
            $locationDetails = processLocationDetails();
            $sellingCount = $locationDetails['sellingCount'];
            $sellings = $locationDetails['sellings'];
            $accounts = $locationDetails['accounts'];

            // Return the view with the required data
            return view("sale", compact(
                "sellings",
                "sellingCount",
                "accounts",
                "status",
            ));
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function vat($net)
    {
        $answer = round(((18 / 100) * $net));
        return $answer;
    }

    public function receipt(Request $request, $id)
    {
        // Incoming data from POST
        $items = $request->all();

        // Get sale by id
        $sale = Sale::with('customer', 'user', 'selling.location')->find($id);

        // Prepare profile info if available (or fallback)
        $profileData = BusinessProfile::where('selling_id', $sale->selling_id)->first();

        $status = $request->status ?? 'sale';

        $latestCustomerDebt = $request->latestCustomerDebt ?? null;

        return view('receipt', compact('items', 'profileData', 'status', 'latestCustomerDebt', 'sale'));
    }

    public function tempReceipt(Request $request, $id)
    {
        dd($request->all());
        // Fetch the sale by ID including all related data needed for the print view
        $sale = Sale::with(['customer', 'soldItems', 'user'])->findOrFail($id);
        $profileData = BusinessProfile::where('selling_id', $sale->selling_id)->with('selling.location')->first();

        // If you need to gather other form details, you fetch them here as well, or calculate any needed values

        // Return the view for printing with all necessary data
        return view('sales.print_receipt', compact('sale', 'profileData'));
    }


    public function splitBillView($locationId, $tableId, $waiterId)
    {
        try {
            # Get location details
            $locationDetails = processLocationDetails();
            $sellings = $locationDetails['sellings'];

            $waiterData = Waiter::find($waiterId);
            $tableData = Table::find($tableId);

            $categories = ProductCategory::where("name", "!=", null)
                ->Where("name", "!=", "Service")
                ->Where("name", "!=", "Ingredient")
                ->get();

            $businessProfile = BusinessProfile::where('selling_id', $waiterData->selling_id)->first();

            return view('split-bill', compact(
                'sellings',
                'locationId',
                'tableData',
                'waiterData',
                'categories',
                "sellings",
                "businessProfile"
            ));
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function splitBill(Request $request)
    {
        $profileData = BusinessProfile::select(
            'name',
            'phone',
            'address',
        )->first();
        $saleId = [];

        $waiterData = Waiter::where('name', $request->waiter)->first();
        $tableData = Table::where('name', $request->table)->first();
        $order_total = $request->total;

        // Initialize an empty associative array where each key is an array itself to store multiple values for each key.
        $orderArray = [
            "location" => array(),
            "id" => array(),
            "product" => array(),
            "unit_id" => array(),
            "price" => array(),
            "qty" => array(),
            "amount" => array(),
        ];

        // Function to check if the combination of "location" and "id" exists in the arrays.
        // If it exists, increment the quantity and amount for that product.
        // If it doesn't exist, add a new entry with the provided data to each key array.
        function incrementProductQuantity(&$orderArray, $location, $id, $product, $unit, $qty, $price, $amount)
        {
            // Check if the combination of "location" and "id" exists in the arrays.
            $index = array_search([$location, $id], array_map(null, $orderArray["location"], $orderArray["id"]));

            if ($index !== false) {
                // If the combination of "location" and "id" already exists, increment the quantity and amount.
                $orderArray["qty"][$index] += $qty;
                $orderArray["amount"][$index] += $amount;
            } else {
                // If the combination of "location" and "id" is not found, add new data to each key array.
                $orderArray["location"][] = $location;
                $orderArray["id"][] = $id;
                $orderArray["product"][] = $product;
                $orderArray["unit_id"][] = $unit;
                $orderArray["price"][] = $price;
                $orderArray["qty"][] = $qty;
                $orderArray["amount"][] = $amount;
            }
        }

        // Loop through the input data to populate the $orderArray using the incrementProductQuantity() function.
        for ($i = 0; $i < count($request->id); $i++) {
            // Function call.
            incrementProductQuantity(
                $orderArray,
                $request->area[$i],
                $request->id[$i],
                $request->product[$i],
                $request->unit[$i],
                $request->qty[$i],
                $request->price[$i],
                $request->amount[$i]
            );
        }

        $dateTime = date("Y-m-d H:i:s"); // Format: YYYY-MM-DD HH:MM:SS
        $randomNumber = rand(1, 1000); // generating random numbers between 1-10000;
        $status = "split";
        $role = auth()->user()->role;

        return view("print-bill", compact(
            "role",
            "dateTime",
            "profileData",
            "orderArray",
            "order_total",
            "waiterData",
            "tableData",
            "randomNumber",
            "status"
        ));
    }

    public function returnView()
    {
        return response()->json(['Sorry' => 'This functionality is still under maintenance!!'], 500);

        try {
            # Get location details
            $locationDetails = processLocationDetails();
            $sellingCount = $locationDetails["sellingCount"];
            $sellings = $locationDetails["sellings"];

            return view("sale-return", compact("sellings", "sellingCount"));
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function insertNewCustomerDebt(Request $request, $sale, $customerId, $sellingId)
    {
        // Format the total, discount and paid amounts
        $total = (int)str_replace(',', '', $request->total);
        $discount = (int)str_replace(',', '', $request->discount);
        $paid = (int)str_replace(',', '', $request->paid);
        $due = (int)str_replace(',', '', $request->due);

        // Check if the customer has existing debts at the same location and update the debt amount.
        $existingDebt = Debts::where('customer_id', $customerId)
            ->where('selling_id', $sellingId)
            ->latest()
            ->first();

        if ($request->type == "credit") {
            $debt = new Debts();
            $debt->date = $request->date;
            $debt->sale_id = $sale->id;
            $debt->customer_id = $customerId;
            $debt->amount = $total;
            $debt->discount = $discount;
            $debt->paid = $paid;
            $debt->due = ($existingDebt ? $existingDebt->due : 0) + ($total - $discount - $paid);
            $debt->selling_id = $sellingId;
            $debt->save();
        } elseif ($request->type == "cash") {
            $debt = new Debts();
            $debt->date = $request->date;
            $debt->sale_id = $sale->id;
            $debt->customer_id = $customerId;
            $debt->paid = abs($due);
            $debt->due = ($existingDebt ? $existingDebt->due : 0) - (abs($due));
            $debt->selling_id = $sellingId;
            $debt->save();
        }
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request, $status)
    {
        DB::beginTransaction();

        $saleSession = null;
        $soldItemSession = null;
        $sale = collect();

        // If editing an order
        if ($status == "edit-order") {
            $saleSession = Session::get('salesDetails');
            $soldItemSession = Session::get('soldItems');

            if ($saleSession->status == 'credit') {
                Debts::where('selling_id', $saleSession->selling_id)
                    ->where("customer_id", $saleSession->customer_id)
                    ->where("sale_id", $saleSession->id)
                    ->delete();

                updateCustomerDebts($saleSession->selling_id, $saleSession->customer_id);
            }
        }

        $sellingId = Selling::where('location_id', $request->location)->value('id');
        $profileData = BusinessProfile::where('selling_id', $sellingId)->first();

        // Handle customer
        $customerId = null;
        if ($request->customer) {
            $customerData = Customer::where("name", $request->customer)
                ->where("selling_id", $sellingId)
                ->first();

            if ($customerData) {
                $customerId = $customerData->id;
            } else {
                $newCustomer = new Customer();
                $newCustomer->name = ucwords($request->customer);
                $newCustomer->address = ucwords($request->address);
                $newCustomer->phone = $request->phone ?? "Unknown";
                $newCustomer->limit = 0;
                $newCustomer->opening_balance = 0;
                $newCustomer->selling_id = $sellingId;
                $newCustomer->save();
                $customerId = $newCustomer->id;
            }
        }

        // Determine sale status
        $saleStatus = $request->type == 'cash' ? 'cash' : 'credit';

        // Clean numeric inputs
        $total = (int) str_replace(',', '', $request->total);
        $discount = (int) str_replace(',', '', $request->discount);
        $paid = (int) str_replace(',', '', $request->paid);
        $net = (int) str_replace(',', '', $request->net);
        $due = (int) str_replace(',', '', $request->due);

        // Determine if royalty applies (every 7th sale)
        $applyRoyalty = false;
        $royaltyDiscountPercent = 0;

        if ($profileData->royalty_enabled && $customerId) {
            $customerSaleCount = Sale::where('customer_id', $customerId)
                ->where('selling_id', $sellingId)
                ->count();

            $nextSaleNumber = $customerSaleCount + 1;
            if ($nextSaleNumber % 7 === 0) {
                $applyRoyalty = true;
                $royaltyDiscountPercent = $profileData->royalty_discount;
            }
        }

        // Create or update sale
        if ($status == "edit-order") {
            $sale = Sale::find($saleSession->id);
            $sale->total = $total;
            $sale->discount = $discount;
            $sale->net = $net;
            $sale->paid = $paid;
            $sale->due = $due;
            $sale->account_id = $request->account;
            $sale->customer_id = $customerId;
            $sale->status = $request->type;
            $sale->selling_id = $sellingId;
            $sale->update();
        } else {
            $saleData = [
                'date' => $request->date,
                'user_id' => auth()->id(),
                'selling_id' => $sellingId,
                'total' => $total,
                'discount' => $discount,
                'paid' => $paid,
                'account_id' => $request->account,
                'salesmen_id' => $request->salesman ?: null,
                'customer_id' => $customerId
            ];

            if ($profileData->vat == "Inclusive") {
                $net = ($total + $this->vat($total)) - $discount;
                $saleData['vat'] = $this->vat($request->total);
                $saleData['net'] = $net;
                $saleData['due'] = $net - $paid;
            } else {
                $saleData['vat'] = 0;
                $saleData['net'] = $net;
                $saleData['due'] = $due;
            }

            $sale = new Sale($saleData);
            $sale->status = $saleStatus;
            $sale->save();
        }

        // Handle customer debts
        if ($saleStatus == 'credit' && $customerId != null) {
            if ($status == 'edit-order') {
                $existingDebt = Debts::where('customer_id', $saleSession->customer_id)
                    ->where('selling_id', $saleSession->selling_id)
                    ->where('sale_id', $saleSession->id)
                    ->first();

                if ($existingDebt) {
                    $existingDebt->date = $request->date;
                    $existingDebt->sale_id = $sale->id;
                    $existingDebt->customer_id = $customerId;
                    $existingDebt->amount = $total;
                    $existingDebt->discount = $discount;
                    $existingDebt->paid = $paid;
                    $existingDebt->due = $due;
                    $existingDebt->selling_id = $sellingId;
                    $existingDebt->update();

                    updateCustomerDebts($sellingId, $customerId);
                } else {
                    $this->insertNewCustomerDebt($request, $sale, $customerId, $sellingId);
                }
            } else {
                $this->insertNewCustomerDebt($request, $sale, $customerId, $sellingId);
            }
        } elseif ($saleStatus == 'cash' && $customerId != null && $due < 0) {
            $this->insertNewCustomerDebt($request, $sale, $customerId, $sellingId);
        }

        // Reverse previous stock for edit
        if ($status == "edit-order") {
            foreach ($soldItemSession as $item) {
                $batchNo = $item->batch_no ?? null;

                recordStockMovement(
                    $request->date,          // $date
                    auth()->id(),            // $userId
                    $item->product_id,       // $productId
                    $request->location,      // $locationId
                    "Edit",                  // $reason
                    "In",                    // $status
                    $item->qty * $item->cont, // $qty
                    $batchNo,                // $batchNo (optional)
                    null                     // $expDate (optional, null here)
                );

                SoldItem::where('sale_id', $saleSession->id)
                    ->where('created_at', $item->created_at)
                    ->delete();
            }
        }

        // Process sold items
        $totalItemDiscount = 0;

        for ($i = 0; $i < count($request->id); ++$i) {
            $productId = $request->id[$i];
            $qty = $request->qty[$i];
            $cont = $request->cont[$i];
            $price = (int) str_replace(',', '', $request->price[$i]);
            $amount = $qty * $cont * $price;
            $disc = (int) str_replace(',', '', $request->disc[$i]);

            $itemDiscount = $applyRoyalty ? ($amount * $royaltyDiscountPercent) / 100 : $disc;
            $totalItemDiscount += $itemDiscount;

            $batchNo = $request->batch_no[$i] ?? null; // <-- optional batch number

            $soldItem = new SoldItem([
                'sale_id' => $sale->id,
                'product_id' => $productId,
                'unit_id' => $request->unit[$i],
                'cont' => $cont,
                'price' => $price,
                'qty' => $qty,
                'amount' => $amount,
                'discount' => $itemDiscount,
                'batch_no' => $batchNo
            ]);

            $soldItem->save();

            // Record stock movement per batch if provided
            recordStockMovement(
                $request->date,          // $date
                auth()->id(),            // $userId
                $productId,              // $productId
                $request->location,      // $locationId
                "Sale",                  // $reason
                "Out",                   // $status
                $qty * $cont,            // $qty
                $batchNo,                // $batchNo (optional)
                null                // $expDate (optional)
            );
        }

        // Apply total royalty discount if applicable
        if ($applyRoyalty) {
            $sale->discount += $totalItemDiscount;
            $sale->net -= $totalItemDiscount;
            $sale->due = $sale->net - $sale->paid;
            $sale->update();
        }

        // Handle payment transactions
        if ($paid != 0 && $request->account != 0) {
            $account = Account::find($request->account);
            $transactionAmount = $paid;

            if ($status == 'edit-order') {
                $transactionAmount = ($paid > $saleSession->paid) ? $paid - $saleSession->paid : $saleSession->paid - $paid;
            }

            recordTransaction(
                $request->date,
                "Sale",
                "In",
                $transactionAmount,
                $account->id,
                auth()->id(),
                $sellingId
            );
        }

        DB::commit();

        session()->forget(['profomaData', 'profomaItems', 'customerData', 'salesDetails', 'soldItems']);

        $profileData = BusinessProfile::where("selling_id", $sellingId)
            ->with('selling.location')
            ->first();

        $time = $sale->created_at->format('H:i:s');
        $latestCustomerDebt = Debts::where("customer_id", $customerId)
            ->where("selling_id", $sellingId)
            ->with("selling.location")
            ->latest()
            ->first();

        $items = SoldItem::where('sale_id', $sale->id)
            ->with("unit", "product")
            ->get();

        $sum = 0;

        if ($profileData->receipt == "Yes") {
            return view('receipt', compact(
                "profileData",
                "sale",
                "time",
                "items",
                "sum",
                "latestCustomerDebt",
                "status"
            ));
        } else {
            return redirect()->route('sale.index', ['id' => 0, 'status' => 'sale'])
                ->with('success', 'Sales posted.');
        }
    }


    // Function to process loaded vehicle.
    private function processLoadedVehicle($request, $i, $saleId = null)
    {
        $loaded_vehicle_exists = LoadVehicle::where('location_id', $request->location)->exists();

        if ($request->locationstatus == "dynamic" && $loaded_vehicle_exists) {
            $base_unit = UnitAssigned::where("product_id", $request->id[$i])
                ->orderBy("unit_cont", 'asc')
                ->first();

            $loaded_item = LoadedItems::where("product_id", $request->id[$i])->first();
            $loaded_item->qty -= $request->qty[$i] * $request->cont[$i];
            $loaded_item->amount -= $request->qty[$i] * $request->cont[$i] * $base_unit->selling;
            $loaded_item->update();

            if ($loaded_item->qty == 0 && $loaded_item->amount == 0) {
                $loaded_item->delete();
            }

            $loaded_vehicle = LoadVehicle::where('location_id', $request->location)->first();
            $loaded_vehicle->total -= $request->qty[$i] * $request->cont[$i] * $base_unit->selling;
            $loaded_vehicle->update();

            if ($loaded_vehicle->total == 0) {
                $loaded_vehicle->delete();
            }

            // ✅ Record the stock movement here
            recordStockMovement(
                $request->date,
                auth()->id(),
                $request->id[$i],
                $request->location,
                "Vehicle Load Adjustment",
                "Out",
                $request->qty[$i] * $request->cont[$i],
                $saleId,
                'vehicle_load'
            );
        }
    }

    // Function to process a single sold item and update stock movement.
    private function processSoldItem($request, $sale, $i, &$sellingId)
    {
        $item_price = str_replace(',', '', $request->price[$i]);
        $item_amount = str_replace(',', '', $request->amount[$i]);
        $item_disc = str_replace(',', '', $request->disc[$i]);

        $items = new SoldItem();
        $items->sale_id = $sale->id;
        $items->product_id = $request->id[$i];
        $items->unit_id = $request->unit[$i];
        $items->cont = $request->cont[$i];
        $items->price = $item_price;
        $items->qty = $request->qty[$i];
        $items->amount = $item_amount;
        $items->discount = (int)$item_disc;
        $items->save();

        $product = Product::find($request->id[$i]);

        if (!$product) {
            return; # or continue if inside a loop
        }

        $categoryName = optional($product->category)->name;

        if ($categoryName === 'Service') {
            $this->processServiceProduct($request, $product, $i, $sellingId);
        } else {
            if ($request->status === 'dynamic') {
                $this->processLoadedVehicle($request, $i, $sale->id);
            } else {
                // Stock movement helper function.
                recordStockMovement(
                    $request->date,
                    auth()->id(),
                    $request->id[$i],
                    $request->location,
                    "Sale",
                    "Out",
                    $request->qty[$i] * $request->cont[$i],
                    $sale->id,
                    'sale'
                );
            }
        }
    }

    // Function to process service products.
    function processServiceProduct($request, $product, $i, $sale)
    {
        $items = ServiceProduct::where("product_id", $product->id)->get();

        if ($items->isEmpty()) {
            return; # or continue if inside a loop
        }

        foreach ($items as $item) {
            $cont = UnitAssigned::where('product_id', $product->id)
                ->where('unit_id', $item->unit_id)->first()->value('unit_cont');

            $answer = $item->qty * $cont * $request->qty[$i];

            // Stock movement helper function.
            recordStockMovement(
                $request->date,
                auth()->id(),
                $item->product_id,
                $item->location_id,
                "Service Product Component",
                "Out",
                $answer, // total qty being consumed
                $sale->id,
                'service_sale'
            );
        }
    }

    public function cancel($id)
    {
        DB::beginTransaction();

        try {
            $sale = Sale::with(['soldItems', 'customer', 'account', 'selling'])->findOrFail($id);

            // 1. Reverse stock (with batch_no)
            foreach ($sale->soldItems as $item) {
                $stockQty = $item->qty * $item->cont;

                $batchNo = $item->batch_no ?? null;
                $expDate = $item->exp_date ?? null; // ✅ if you are tracking expiry dates

                recordStockMovement(
                    now()->toDateString(),    // date
                    auth()->id(),             // user_id
                    $item->product_id,        // product_id
                    $sale->selling->location_id, // location_id
                    "Sale Cancellation",      // reason
                    "In",                     // status
                    $stockQty,                // qty
                    $batchNo,                 // ✅ batch number restored
                    $expDate                  // ✅ expiry date restored
                );
            }

            // 2. Reverse debts
            Debts::where('sale_id', $sale->id)->delete();
            updateCustomerDebts($sale->selling_id, $sale->customer_id);

            // 3. Reverse account transactions
            if ($sale->paid > 0 && $sale->account_id) {
                recordTransaction(
                    now()->toDateString(),    // date
                    "Sale Cancellation",      // reason
                    "Out",                    // status
                    $sale->paid,              // amount
                    $sale->account_id,        // account_id
                    auth()->id(),             // user_id
                    $sale->selling_id         // selling_id (to keep consistent with your bulkDelete/store)
                );
            }

            // 4. Delete sold items & sale
            SoldItem::where('sale_id', $sale->id)->delete();
            $sale->delete();

            DB::commit();

            return response()->json(['message' => 'Sale cancelled successfully.']);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error("Sale cancel error: " . $e->getMessage());
            return response()->json(['error' => 'Failed to cancel sale.'], 500);
        }
    }


    public function salesReturn(Request $request)
    {
        $return = new SaleReturn();
        $return->date = $request->date;
        $return->user_id = auth()->user()->id;
        $return->total = $request->total;
        $return->account_id = $request->account;
        $return->customer_id = $request->customer;
        $return->location_id = $request->location;
        $return->save();

        for ($i = 0; $i < count($request->id); ++$i) {
            $items = new SalesReturnItem();
            $items->return_id = $return->id;
            $items->product_id = $request->id[$i];
            $items->unit_id = $request->unit[$i];
            $items->cont = $request->cont[$i];
            $items->price = $request->price[$i];
            $items->qty = $request->qty[$i];
            $items->amount = $request->amount[$i];
            $items->save();

            $qtyCont = $request->qty[$i] * $request->cont[$i];

            if (Stock::where("product_id", $request->id[$i])
                ->where("location_id", $request->area)
                ->exists()
            ) {
                $stock = Stock::where("product_id", $request->id[$i])
                    ->where("location_id", $request->area)
                    ->first();

                $movement = new StockMovement();
                $movement->date = $request->date;
                $movement->user_id = auth()->user()->id;
                $movement->product_id =  $request->id[$i];
                $movement->reason = "Sales Return";
                $movement->status = "In";
                $movement->qty = $qtyCont;
                $movement->before = $stock->qty;
                $movement->after = $stock->qty + $qtyCont;
                $movement->location_id = $request->location;
                $movement->save();

                $stock->qty += $qtyCont;
                $stock->save();
            } else {
                // Get the old product from products table
                $oldProduct = Product::where("id", $request->id[$i])
                    ->where("location_id", $request->location)
                    ->first();
                // Create new product in the respective location with old products details.
                $newProduct = new Product();
                $newProduct->barcode = $oldProduct->barcode;
                $newProduct->image = $oldProduct->image;
                $newProduct->barcode = $oldProduct->barcode;
                $newProduct->category_id = $oldProduct->category_id;
                $newProduct->buying = $oldProduct->buying;
                $newProduct->supplier_id = $oldProduct->supplier_id;
                $newProduct->location_id = $request->location;
                $newProduct->save();

                // Get the old product units
                $oldUnits = UnitAssigned::where("Product_id", $request->id[$i])
                    ->where("location_id", $request->location)
                    ->get();
                // Loop through the old units
                foreach ($oldUnits as $oldUnit) {
                    //create new units in  units assigned table.
                    $newUnit = new UnitAssigned();
                    $newUnit->product_id = $oldUnit->product_id;
                    $newUnit->unit_id = $oldUnit->unit_id;
                    $newUnit->unit_cont = $oldUnit->unit_cont;
                    $newUnit->selling = $oldUnit->selling;
                    $newUnit->location_id = $request->location;
                    $newUnit->save();
                }

                // Get the old stock details in stock table.
                $oldStock = stock::where("product_id", $request->id[$i])
                    ->where("location_id", $request->location)
                    ->first();

                // Create a stock item and add it to the stock items array
                $newStock = new Stock();
                $newStock->product_id = $request->id[$i];
                $newStock->location_id = $request->location;
                $newStock->unit_id = $oldStock->unit_id;
                $newStock->min_cont = $oldStock->min_cont;
                $newStock->min_qty = $oldStock->min_qty;
                $newStock->qty = $request->qty[$i] * $request->cont[$i];
                $newStock->exp = $oldStock->exp;
                $newStock->exp_not = $oldStock->exp_not;
                $newStock->save();
            }
        }

        $account = Account::find($request->account);

        $account->balance -= $request->total;
        $account->update();

        $transaction = new Transaction();
        $transaction->date = $request->date;
        $transaction->user_id = auth()->user()->id;
        $transaction->reason = "Sale return";
        $transaction->status = "Out";
        $transaction->amount = $request->total;
        $transaction->before = $account->balance + $request->total;
        $transaction->after = $account->balance;
        $transaction->account_id = $account->id;
        $transaction->location_id = $request->location;
        $transaction->save();

        return redirect()->route('sales.return.view')->with("success", "Posted successfully");
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function SalesReceipt(Request $request)
    {
        try {
            # Get location details
            $locationDetails = processLocationDetails();
            $sellingCount = $locationDetails['sellingCount'];
            $sellings = $locationDetails['sellings'];

            // ✅ Default sellingId to authenticated user's selling
            $sellingId = auth()->user()->selling_id ?? null;

            // ✅ Override sellingId if a specific area is selected
            if ($request->filled('area') && $request->area != 0) {
                $sellingId = Selling::where('location_id', $request->area)
                    ->value('id');
            }

            // Get current date
            $currentDate = date('Y-m-d');

            // ✅ Default type to "today" if not provided
            $type = $request->type ?? 'today';

            // ✅ Default start and end to current date
            $start = $request->start ?? $currentDate;
            $end = $request->end ?? $currentDate;

            // Start sale query
            $saleQuery = Sale::query();

            // Date filter
            if ($type === "today") {
                $saleQuery->where("date", $currentDate);
            } elseif ($type === "set date") {
                $saleQuery->where("date", $start);
            } else {
                $saleQuery->whereBetween("date", [$start, $end]);
            }

            // Selling filter (skip if sellingId is 'all')
            if ($sellingId && $sellingId !== 'all') {
                $saleQuery->where("selling_id", $sellingId);
            }

            // Get the filtered sales records along wiht its respective relations
            $sales = $saleQuery
                ->with([
                    "customer",
                    "selling.location",
                    "user",
                    "account" => function ($query) {
                        $query->withTrashed();
                    }
                ])
                ->orderBy('created_at', 'desc')
                ->get();

            return view("sales-receipt", compact("sales", "sellings", "sellingCount"));
        } catch (Exception $e) {
            Log::error($e->getMessage());
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function getSpecificSale(Request $request, $id)
    {
        try {
            // Retrieve the specific sale and its associated sold items
            $sales = DB::table("sales")
                ->join("sold_items", "sold_items.sale_id", "=", "sales.id")
                ->where("sales.id", $id)
                ->get();

            // Return the specific-sale view, passing the sales data
            return view("specific-sale", compact('sales'));
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        //
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function bulkDelete(Request $request)
    {
        DB::beginTransaction();

        try {
            $ids = $request->ids;

            foreach ($ids as $id) {
                $item = SoldItem::find($id);
                if (!$item) continue;

                $sale = Sale::with('selling')->find($item->sale_id);
                if (!$sale) continue;

                $sellingId = $sale->selling_id;
                $customerId = $sale->customer_id;
                $productId = $item->product_id;

                // 1. Update debts
                $debt = Debts::where('sale_id', $sale->id)
                    ->where('customer_id', $customerId)
                    ->first();

                if ($debt) {
                    $debt->amount -= $item->amount;
                    $debt->discount -= $item->discount ?? 0;
                    $debt->due -= ($item->amount - ($item->discount ?? 0));
                    $debt->save();

                    if ($debt->amount <= 0) {
                        $debt->delete();
                    }
                }

                updateCustomerDebts($sellingId, $customerId);

                // 2. Reverse stock (with batch_no)
                $product = Product::find($productId);
                if (!$product) continue;

                $locationId = $product->location_id;
                $category = ProductCategory::find($product->category_id);

                $batchNo = $item->batch_no ?? null;

                if ($category && $category->name === "Service") {
                    $services = ServiceProduct::where("product_id", $productId)->get();

                    foreach ($services as $service) {
                        $unitCont = UnitAssigned::where("unit_id", $service->unit_id)->value('unit_cont') ?? 1;
                        $restoredQty = $service->qty * $unitCont * $item->qty;

                        $stock = Stock::where("product_id", $productId)
                            ->where("location_id", $locationId)
                            ->when($batchNo, function ($query) use ($batchNo) {
                                $query->where("batch_no", $batchNo);
                            })
                            ->first();

                        if ($stock) {
                            recordStockMovement(
                                now()->toDateString(),
                                auth()->id(),
                                $productId,
                                $locationId,
                                'Sale deleted',
                                'In',
                                $restoredQty,
                                $batchNo,   // ✅ pass batch number
                                null
                            );
                        }
                    }
                } else {
                    $stock = Stock::where("product_id", $productId)
                        ->where("location_id", $locationId)
                        ->when($batchNo, function ($query) use ($batchNo) {
                            $query->where("batch_no", $batchNo);
                        })
                        ->first();

                    if ($stock) {
                        $restoredQty = $item->qty * $item->cont;

                        recordStockMovement(
                            now()->toDateString(),
                            auth()->id(),
                            $productId,
                            $locationId,
                            'Sale deleted',
                            'In',
                            $restoredQty,
                            $batchNo,   // ✅ pass batch number
                            null
                        );
                    }
                }

                // 3. Account / advance handling
                if ($sale->account_id == 0) {
                    $customer = Customer::find($customerId);
                    if ($customer) {
                        $customer->increment('advance', $item->amount - ($item->discount ?? 0));
                    }
                } else {
                    $amount = $item->amount - ($item->discount ?? 0);
                    $account = Account::find($sale->account_id);
                    if ($account) {
                        recordTransaction(
                            now()->toDateString(),
                            'Sale deleted',
                            'Out',
                            $amount,
                            $account->id,
                            auth()->id(),
                            $sellingId
                        );
                    }
                }

                // 4. Update sale totals
                $sale->total -= $item->amount;
                $sale->discount -= $item->discount ?? 0;
                $sale->net -= ($item->amount - ($item->discount ?? 0));
                $sale->due -= ($item->amount - ($item->discount ?? 0));

                if ($sale->status === "cash") {
                    $sale->paid -= ($item->amount - ($item->discount ?? 0));
                } elseif ($sale->status === "credit") {
                    $sale->paid = max(0, $sale->paid - ($item->amount - ($item->discount ?? 0)));
                }

                $sale->save();

                // 5. Delete sold item
                $item->delete();

                // 6. Delete sale if no items left
                if (SoldItem::where("sale_id", $sale->id)->count() === 0) {
                    $sale->delete();
                }
            }

            DB::commit();
            return response()->json('success');
        } catch (Exception $e) {
            DB::rollBack();
            Log::error($e->getMessage());
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }



    public function deleteSoldItems(Request $request)
    {
        try {
            $ids = $request->ids;
            $profile = BusinessProfile::select("vat")->first();

            // Fetch sold items in chunks to optimize memory usage
            SoldItem::whereIn('id', $ids)->chunk(100, function ($soldItems) use ($profile) {

                // Get the sale IDs associated with the sold items
                $saleIds = $soldItems->pluck('sale_id')->toArray();

                // Fetch all sales related to the sold items
                $sales = Sale::whereIn('id', $saleIds)->get();

                // Count the number of sold items for each sale
                $saleItemsCount = SoldItem::whereIn('sale_id', $saleIds)->count();

                foreach ($soldItems as $soldItem) {
                    $sale = $sales->where('id', $soldItem->sale_id)->first();
                    $account = Account::find($sale->account_id);

                    // Create a transaction for recording cash movement
                    $transaction = new Transaction();
                    $transaction->date = date('Y-m-d');
                    $transaction->user = auth()->user()->name;
                    $transaction->reason = "Sale deleted";
                    $transaction->status = "Out";
                    $transaction->amount = -$soldItem->amount + $soldItem->discount;
                    $transaction->before = $account->balance;
                    $transaction->after = $account->balance - $soldItem->amount + $soldItem->discount;
                    $transaction->account_id = $account->id;
                    $transaction->location_id = $sale->location_id;
                    $transaction->save();

                    // Update the account balance
                    $account->decrement('balance', $soldItem->amount - $soldItem->discount);

                    $product = Product::where("product", $soldItem->product)->first();

                    if ($product->category == "service") {
                        // Handle service products
                        $serviceItems = ServiceProduct::where("service", $product->product)->get();

                        foreach ($serviceItems as $serviceItem) {
                            $answer = $serviceItem->qty * $serviceItem->cont * $soldItem->qty;
                            $stock = Stock::where("product", $serviceItem->product)
                                ->where("location_id", $sale->location_id)
                                ->first();

                            // Record stock movement for service product
                            $movement = new StockMovement();
                            $movement->date = date('Y-m-d');
                            $movement->user = auth()->user()->name;
                            $movement->product = $soldItem->product;
                            $movement->reason = "Sale deleted";
                            $movement->status = "In";
                            $movement->qty = $soldItem->qty * $soldItem->cont;
                            $movement->before = $stock->qty;
                            $movement->after = $stock->qty + $answer;
                            $movement->location_id = $sale->location_id;
                            $movement->save();

                            // Update stock quantity
                            $stock->increment('qty', $answer);
                        }
                    } else {
                        // Handle normal products
                        $stock = Stock::where("product", $soldItem->product)
                            ->where("location_id", $sale->location_id)
                            ->first();

                        // Record stock movement for normal product
                        $movement = new StockMovement();
                        $movement->date = date('Y-m-d');
                        $movement->user = auth()->user()->name;
                        $movement->product = $soldItem->product;
                        $movement->reason = "Sale deleted";
                        $movement->status = "In";
                        $movement->qty = $soldItem->qty * $soldItem->cont;
                        $movement->before = $stock->qty;
                        $movement->after = $stock->qty + ($soldItem->qty * $soldItem->cont);
                        $movement->location_id = $sale->location_id;
                        $movement->save();

                        // Update stock quantity
                        $stock->increment('qty', $soldItem->qty * $soldItem->cont);
                    }

                    // Include VAT if it's enabled
                    if ($profile->vat == "Inclusive") {
                        $sale->total -= $soldItem->amount - $soldItem->discount + $this->vat($soldItem->amount - $soldItem->discount);
                        $sale->vat -= $this->vat($soldItem->amount - $soldItem->discount);
                    } else {
                        // Don't include VAT if it's disabled
                        $sale->net = $sale->total - $soldItem->amount - $soldItem->discount + $sale->transport;
                        $sale->total -= $soldItem->amount - $soldItem->discount;
                    }

                    // Update the sale
                    $sale->update();

                    // Delete the sold item
                    $soldItem->delete();

                    // Check if the sold item table is empty for the sale
                    if ($saleItemsCount === 1) {
                        // If it is, delete the sale
                        $sale->delete();
                    }
                }
            });

            return response()->json([]);
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function changeDate(Request $request)
    {
        try {
            // Start a new database transaction
            DB::beginTransaction();

            // Initiate variables
            $ids = $request->get("ids");
            $newdate = $request->get("newdate");

            // Iterate over the sold items with the given IDs
            SoldItem::whereIn('id', $ids)->each(function ($item) use ($newdate) {
                // Find the sale associated with the current sold item
                $sale = Sale::find($item->sale_id);

                // Update the date of the sale
                $sale->date = $newdate;
                $sale->update();
            });

            // Commit the changes to the database
            DB::commit();

            return response()->json([]);
        } catch (Exception $e) {
            // Rollback the changes if anything goes wrong
            DB::rollBack();

            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function payCustomerDebt(Request $request, $id)
    {
        DB::beginTransaction();
        try {
            // Fetch the sale with the given ID
            $sale = Sale::findOrFail($id);

            // Initiate variables
            $sellingId = $sale->selling_id;
            $customerId = $request->input('customer-id');
            $accountId = $request->input('account');
            $paid = (int)str_replace(',', '', $request->input('paid'));

            // Retrieve the latest debt for the provided customer and location.
            $currentDebt = Debts::where("customer_id", $customerId)
                ->where("selling_id", $sellingId)
                ->latest()
                ->first();

            // Check if the current_debt is found or not, then return with an error message.
            if ($currentDebt === null) {
                return redirect()->back()->with('error', 'Debt not found.');
            }

            // Record the payment in the Debts table for the provided customer.
            $debt = new Debts();
            $debt->date = $request->date;
            $debt->customer_id = $customerId;
            $debt->paid = $paid;
            $debt->account_id = $accountId;
            $debt->due =  $currentDebt->due - $paid;
            $debt->selling_id = $sellingId;
            $debt->save();

            // Update the paid and due columns of the sale
            $sale->paid += $request->paid;
            $sale->due -= $request->paid;
            $sale->save();

            // Record the transaction in the Transactions table
            recordTransaction(
                $request->date,
                "Sale",
                "In",
                $paid,
                $accountId,
                auth()->id(),
                $sellingId
            );

            DB::commit();
            return back()->with("success", "Payment successfully posted");
        } catch (Exception $e) {
            DB::rollBack();
            Log::error($e->getMessage());
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function payCustomerDebtHistory(Request $request, $customerId)
    {
        $validated = $request->validate([
            'account' => 'required|exists:accounts,id',
            'paid' => 'required|min:1',
            'date' => 'required|date',
        ]);

        $paid = (int)str_replace(',', '', $validated['paid']);

        DB::beginTransaction();
        try {
            $customer = Customer::find($customerId);
            if (!$customer) {
                return back()->with('error', 'Customer not found.');
            }

            $sellingId = $customer->selling_id;
            $accountId = $validated['account'];

            $sale = new Sale();
            $sale->payDebt($customerId, $paid, $sellingId);

            $currentDebt = Debts::where("customer_id", $customerId)
                ->where("selling_id", $sellingId)
                ->latest()
                ->first();

            if (!$currentDebt) {
                return back()->with('error', 'Debt not found.');
            }

            $debt = new Debts();
            $debt->date = $validated['date'];
            $debt->customer_id = $customerId;
            $debt->paid = $paid;
            $debt->account_id = $accountId;
            $debt->due = max(0, $currentDebt->due - $paid);
            $debt->selling_id = $sellingId;
            $debt->save();

            $account = Account::find($accountId);
            if (!$account) {
                return back()->with('error', 'Account not found.');
            }

            $transaction = new Transaction();
            $transaction->fill([
                'user_id' => auth()->id(),
                'date' => $validated['date'],
                'reason' => "Paid customer debt",
                'status' => "In",
                'amount' => $paid,
                'before' => $account->balance,
                'after' => $account->balance + $paid,
                'account_id' => $accountId,
                'selling_id' => $sellingId,
            ]);
            $transaction->save();

            $account->increment('balance', $paid);

            DB::commit();
            return back()->with("success", "Payment successfully posted.");
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error($e->getMessage());
            return back()->with('error', 'Something went wrong.');
        }
    }


    public function uploadSalesExcel(Request $request)
    {
        try {
            Excel::import(new SalesImport, $request->file);
            return back()->with("success", "Excel uploaded!!");
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function dukamaxSalesUpload(Request $request)
    {
        try {
            Excel::import(new HotelmaxSalesImport, $request->file);
            return back()->with("success", "Excel uploaded!!");
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function dukamaxSolditemsUpload(Request $request)
    {
        try {
            Excel::import(new SoldItemsImport, $request->file);
            return back()->with("success", "Excel uploaded!!");
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function itemsMissingProductId()
    {
        try {
            // Get all items from the sold_items table where product_id column is null.
            $sold_items = SoldItem::whereNull('product_id')->get();

            // Loop through the items.
            foreach ($sold_items as $item) {
                // Find the product based on the item's product name.
                $product = Product::where('product', $item->product)->first();

                if ($product) {
                    // Update the product_id column in the sold_items table.
                    $item->product_id = $product->id;
                    $item->save();
                }
            }

            return response()->json(['message' => 'All products missing product_id in sold_items table updated successfully!!']);
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function editCustomerOrder($id)
    {
        // Find the sale record by id
        $sale = Sale::find($id);

        // Get all sold items related to this sale that have associated product and unit data
        $soldItems = SoldItem::where("sale_id", $id)->with('product', 'unit')->get();

        // Set status to "edit-order"
        $status = "edit-order";

        // Store multiple key-value pairs in the session
        Session::put([
            'salesDetails' => $sale,  // Store sale details in session
            'soldItems' => $soldItems,  // Store sold items in session
            // Add more key-value pairs as needed
        ]);

        // Redirect to the sale index page with sale id and status as parameters
        return redirect()->route('sale.index', ['id' => $id, 'status' => $status]);
    }

    public function cancelSale($id, $status)
    {
        session()->forget(['profomaData', 'profomaItems', 'customerData', 'salesDetails', 'soldItems']);
        return redirect()->route('sale.index', ['id' => $id, 'status' => $status])->with('success', 'Sales cancelled.');
    }

    public function advanceCheker(Request $request)
    {
        # Initiate variable
        $response = false;

        # Remove the commas
        $amount = (int)str_replace(',', '', $request->amount);

        $customer = customer::find($request->customerId);
        if ($amount > $customer->advance) {
            $response = true;
        }

        return response()->json(['response' => $response]);
    }
}
