<?php

namespace App\Http\Controllers;

use App\Models\AssignProduct;
use App\Models\Location;
use App\Models\Product;
use App\Models\ProductCategory;
use App\Models\Selling;
use App\Models\Service;
use App\Models\ServiceProduct;
use App\Models\Stock;
use App\Models\Unit;
use App\Models\UnitAssigned;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class ServiceController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        try {
            # Get location details
            $locationDetails = processLocationDetails();
            $sellingCount = $locationDetails['sellingCount'];
            $sellings = $locationDetails['sellings'];

            # Get  service category data 
            $category = ProductCategory::where('name', 'Service')->first();

            if (!$category) {
                redirect()->back()->with('error', 'Service category not found.');
            }

            $services = Product::where('category_id', $category->id)->with('unit.unit', 'location', 'serviceProduct.unit')->get();

            return view("services", compact("services", "sellings"));
        } catch (Exception $e) {
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function serviceProductView($id)
    {
        try {
            DB::beginTransaction();

            $service = product::with('location', 'unit.unit')->find($id);

            DB::commit();

            return view("service-products", compact("service"));
        } catch (Exception $e) {
            DB::rollback();
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */

    public function store(Request $request)
    {
        // Validate request input
        $request->validate([
            'name' => 'required',
            'price' => 'required',
            'location' => 'required',
        ]);

        try {
            DB::beginTransaction(); // Start transaction

            // Retrieve or create required related records
            $category = ProductCategory::firstOrCreate(['name' => 'Service']);
            $unit = Unit::firstOrCreate(['name' => 'Pc']);
            $locationId = $request->location;

            // Insert product details
            $product = new Product();
            $product->barcode = '';
            $product->product = ucwords($request->name);
            $product->category_id = $category->id;
            $product->buying = 0;
            $product->location_id = $locationId;
            $product->save();

            // Insert stock details
            $stock = new Stock();
            $stock->product_id = $product->id;
            $stock->location_id = $locationId;
            $stock->unit_id = $unit->id;
            $stock->min_cont = 1;
            $stock->min_qty = 0;
            $stock->qty = 0;
            $stock->save();

            // Insert unit assignment details
            $unitAssigned = new UnitAssigned();
            $unitAssigned->product_id = $product->id;
            $unitAssigned->unit_id = $unit->id;
            $unitAssigned->unit_cont = 1;
            $unitAssigned->selling = $request->price;
            $unitAssigned->location_id = $locationId;
            $unitAssigned->save();

            DB::commit(); // Commit transaction

            return back()->with('success', 'Successfully added!!');
        } catch (Exception $e) {
            DB::rollBack(); // Rollback transaction
            Log::error('Service creation failed: ' . $e->getMessage());

            return back()->with('error', 'Something went wrong while adding the service.');
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function assignProducts(Request $request, $id)
    {
        DB::beginTransaction(); // Start a database transaction

        try {
            for ($i = 0; $i < count($request->id); ++$i) {
                $serviceProductCheck = ServiceProduct::where("product_id", $request->id[$i])->exists();

                if ($serviceProductCheck == false) {
                    $data = new ServiceProduct();
                    $data->product_id = $request->id[$i];
                    $data->unit_id = $request->unit[$i];
                    $data->qty = $request->qty[$i];
                    $data->save();
                }
            }

            // Commit the transaction if everything is successful
            DB::commit();

            return redirect()->route('service.index')->with("success", "Successfully assigned!!");
        } catch (Exception $e) {
            // Rollback the transaction in case of an error
            DB::rollback();
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        DB::beginTransaction();

        try {
            $product = Product::find($id);

            if (!$product) {
                redirect()->back()->with('error', "Service not found.");
            }

            $product->product = $request->name;
            $product->location_id = $request->location;
            $product->save();

            $unit = UnitAssigned::where("product_id", $id)->first();

            if (!$unit) {
                redirect()->back()->with('error', "Service unit not found.");
            }

            $unit->selling = $request->price;
            $unit->location_id = $request->location;
            $unit->update();

            DB::commit();

            return back()->with("success", "Successfully updated!!");
        } catch (Exception $e) {
            // Rollback the transaction in case of an error
            DB::rollback();
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        DB::beginTransaction();
        try {
            $product = Product::find($id);

            # If not found skip it
            if (!$product) {
                redirect()->back()->with('success', 'Service not found');
            }

            # Delete product stock data
            Stock::where("product_id", $product->id)->delete();

            # Delete the associated units
            UnitAssigned::where("product_id", $product->id)->delete();

            # Delete the services from products table
            $product->delete();

            DB::commit();

            return back()->with("success", "Successfully deleted!!");
        } catch (Exception $e) {
            // Rollback the transaction in case of an error
            DB::rollback();
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function serviceBulkDelete(Request $request)
    {
        DB::beginTransaction();
        try {
            $ids = $request->ids;

            foreach ($ids as $id) {
                # Find the product by id
                $product = Product::find($id);

                # If not found skip it
                if (!$product) {
                    continue;
                }

                # Delete product stock data
                Stock::where("product_id", $product->id)->delete();

                # Delete the associated units
                UnitAssigned::where("product_id", $product->id)->delete();

                # Delete the services from products table
                $product->delete();

                # Delete the product from service product table
                ServiceProduct::where("product_id", $product->id)->delete();
            }
            DB::commit();

            return response()->json(['success' => 'Deleted successfully'], 200);
        } catch (Exception $e) {
            // Rollback the transaction in case of an error
            DB::rollback();
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function productBulkDelete(Request $request)
    {
        DB::beginTransaction();
        try {

            # Get the submited ids 
            $ids = $request->ids;

            # Loop through each id
            foreach ($ids as $id) {
                # find product and id and delete it from service product.
                ServiceProduct::find($id)->delete();
            }
            DB::commit();

            return response()->json(['success' => 'Deleted successfully'], 200);
        } catch (Exception $e) {
            // Rollback the transaction in case of an error
            DB::rollback();
            // Log the error
            Log::error($e->getMessage());
            // Return a custom error response
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }
}
