<?php

// Adjust the namespace as needed
use App\Models\Debts;
use App\Models\StockMovement;
use App\Models\Account;
use App\Models\Asset;
use App\Models\BusinessProfile;
use App\Models\Creditor;
use App\Models\Transaction;
use App\Models\Selling;
use App\Models\Supplier;
use App\Models\Customer;
use App\Models\Debtor;
use App\Models\Profoma;
use App\Models\User;
use Carbon\Carbon;
use App\Models\License;
use App\Models\Loan;
use App\Models\Location;
use App\Models\Main;
use App\Models\PurchaseOrder;
use App\Models\Stock;
use App\Models\Storage;
use App\Models\SupplierDebts;
use App\Models\Table;
use App\Models\Waiter;
use Illuminate\Support\Facades\DB;

function updateCustomerDebts($sellingId, $customerId)
{
    // Fetch customer debts only once
    $customerDebts = Debts::where('selling_id', $sellingId)
        ->where("customer_id", $customerId)
        ->orderBy('created_at', 'ASC')
        ->get();

    if ($customerDebts->isNotEmpty()) { // Check if there are any debts found for that customer
        $previousDueBalance = null;

        // Re-calculate customer debts.
        foreach ($customerDebts as $debt) {
            if ($previousDueBalance === null) {
                $previousDueBalance = $debt->opening_balance + $debt->amount - $debt->discount - $debt->paid;
                $debt->due = $previousDueBalance;
                $debt->save();
                continue; // Skip the rest of the loop for the first entry.
            }

            $debt->due = $previousDueBalance + $debt->amount - $debt->discount - $debt->paid;
            $debt->save();

            $previousDueBalance = $debt->due;
        }
    }
}

function updateSupplierDebts($sellingId, $supplierId)
{
    DB::beginTransaction();

    try {
        $supplierDebts = SupplierDebts::where('selling_id', $sellingId)
            ->where('supplier_id', $supplierId)
            ->orderBy('created_at', 'ASC')
            ->get();

        if ($supplierDebts->isNotEmpty()) {
            $previousDueBalance = null;

            foreach ($supplierDebts as $debt) {
                $discount = $debt->discount ?? 0;
                $paid = $debt->paid ?? 0;
                $amount = $debt->amount ?? 0;
                $openingBalance = $debt->opening_balance ?? 0;

                if ($previousDueBalance == null) {
                    $newDue = $openingBalance + $amount - $discount - $paid;
                } else {
                    $newDue = $previousDueBalance + $amount - $discount - $paid;
                }

                $debt->update(['due' => $newDue]);

                $previousDueBalance = $newDue;
            }
        }
        DB::commit();
    } catch (\Throwable $e) {
        DB::rollBack();
        throw $e;
    }
}

// Function to process stock movement.
function recordStockMovement($date, $userId, $productId, $locationId, $reason, $status, $qty, $batchNo = null, $expDate = null)
{
    $stock = Stock::where("product_id", $productId)
        ->where("location_id", $locationId)
        ->when($batchNo, function ($query) use ($batchNo) {
            $query->where("batch_no", $batchNo);
        })
        ->first();

    $beforeQty = $stock ? $stock->qty : 0;
    $afterQty = ($status === "Out") ? $beforeQty - $qty : $beforeQty + $qty;

    // Create stock movement record
    StockMovement::create([
        'date' => $date,
        'user_id' => $userId,
        'product_id' => $productId,
        'location_id' => $locationId,
        'reason' => $reason,
        'status' => $status,
        'qty' => $qty,
        'before' => $beforeQty,
        'after' => $afterQty,
        'batch_no' => $batchNo,
        'exp' => $expDate,
    ]);

    if ($stock) {
        $stock->update(['qty' => $afterQty]);
    } else {
        $newStockData = [
            'product_id' => $productId,
            'location_id' => $locationId,
            'min_cont' => $stock->min_cont ?? 1,
            'min_qty' => $stock->min_qty ?? 2,
            'qty' => $afterQty,
        ];
        if ($batchNo) {
            $newStockData['batch_no'] = $batchNo;
            $newStockData['exp_date'] = $expDate;
        }
        Stock::create($newStockData);
    }
}

function processAccountTransaction($request, $saleSession, $status)
{
    // If the account is not "advance", find the account and calculate the amount based on the status
    $account = Account::find($request->account);
    $computedAmount = 0;

    // If the status is "edit-order", compute the difference between the paid and the session paid amount
    if ($status == 'edit-order') {
        $computedAmount = $request->paid - $saleSession->paid;
    } else {
        $computedAmount = $request->paid;
    }

    // Create a new transaction with the appropriate fields
    $transaction = new Transaction([
        'date' => $request->date,
        'user_id' => auth()->user()->id,
        'reason' => "Sale",
        'status' => ($status == 'edit-order') ? "Out" : "In",
        'amount' => $computedAmount,
        'before' => $account->balance,
        'after' => $account->balance + $computedAmount,
        'account_id' => $request->account,
        'selling_id' => $account->selling_id,
    ]);

    // Save the transaction to the database
    $transaction->save();

    // Update the account balance with the computed amount
    $account->update(['balance' => $account->balance + $computedAmount]);
}

function processLocationDetails()
{
    # Start Queries
    $sellingQuery = Selling::query();
    $storageQuery = Storage::query();
    $accountQuery = Account::query();
    $supplierQuery = Supplier::query();
    $customerQuery = Customer::query();
    $profomaQuery = Profoma::query();
    $purchaseOrderQuery = PurchaseOrder::query();
    $customerDebts = Debts::query();
    $supplerDebts = SupplierDebts::query();
    $transactionQuery = Transaction::query();
    $waiterQuery = Waiter::query();
    $tableQuery = Table::query();
    $userQuery = User::query();
    $assetQuery = Asset::query();
    $creditorQuery = Creditor::query();
    $debtorQuery = Debtor::query();
    $loanQuery = Loan::query();

    $userSellingLocation = auth()->user()->selling_id; // Get the logged in user main location.

    # Filler by selling id
    if ($userSellingLocation != "all") {
        $sellingQuery->where("id", $userSellingLocation);
        $storageQuery->where("selling_id", $userSellingLocation);
        $accountQuery->where("selling_id", $userSellingLocation);
        $supplierQuery->where("selling_id", $userSellingLocation);
        $customerQuery->where("selling_id", $userSellingLocation);
        $profomaQuery->where("selling_id", $userSellingLocation);
        $purchaseOrderQuery->where("selling_id", $userSellingLocation);
        $customerDebts->where("selling_id", $userSellingLocation);
        $supplerDebts->where("selling_id", $userSellingLocation);
        $transactionQuery->where("selling_id", $userSellingLocation);
        $waiterQuery->where("selling_id", $userSellingLocation);
        $tableQuery->where("selling_id", $userSellingLocation);
        $userQuery->where("selling_id", $userSellingLocation);
        $assetQuery->where("selling_id", $userSellingLocation);
        $creditorQuery->where("selling_id", $userSellingLocation);
        $loanQuery->where("selling_id", $userSellingLocation);
    }

    # Load relations
    $sellings = $sellingQuery->with('location')->get();
    $sellingCount = $sellings->count();
    $accounts = $accountQuery->get();
    $suppliers = $supplierQuery->with('selling.location')->get();
    $customers = $customerQuery->with('selling.location')->get();
    $profomas = $profomaQuery->with('selling.location')->get();
    $purchaseOrders = $purchaseOrderQuery->with('supplier', 'user', 'selling.location')->get();
    $customerDebts = $customerDebts->with('selling.location')->get();
    $supplerDebts = $supplerDebts->with('selling.location')->get();
    $transactions = $transactionQuery->with('selling.location')->get();
    $waiters = $waiterQuery->with('selling.location')->get();
    $tables = $tableQuery->with('selling.location')->get();
    $storages = $storageQuery->with('location')->get();
    $users = $userQuery->with('selling.location')->get();
    $assets = $assetQuery->with('selling.location')->get();
    $creditors = $creditorQuery->with('selling.location')->get();
    $debtors = $debtorQuery->with('selling.location')->get();
    $loans = $loanQuery->with('selling.location')->get();
    $mains = Main::with('location')->get();
    $all_locations = Location::where('type', '!=', 'main')->get();
    # End Queries
    return [
        'sellingCount' => $sellingCount,
        'sellings' =>  $sellings,
        'accounts' => $accounts,
        'suppliers' => $suppliers,
        'customers' => $customers,
        'profomas' => $profomas,
        'purchaseOrders' => $purchaseOrders,
        'customerDebts' => $customerDebts,
        'supplierDebts' => $supplerDebts,
        'transactions' => $transactions,
        'userSellingLocation' => $userSellingLocation,
        'all_locations' => $all_locations,
        'storages' => $storages,
        'mains' => $mains,
        'users' => $users,
        'waiters' => $waiters,
        'tables' => $tables,
        'assets' => $assets,
        'creditors' => $creditors,
        'debtors' => $debtors,
        'loans' => $loans,
    ];
}

if (!function_exists('getFormattedLicenseRemainingTime')) {
    function getFormattedLicenseRemainingTime()
    {
        $license = License::latest()->first(); // Adjust if using multiple licenses

        if (!$license) {
            return null;
        }

        $now = Carbon::now();
        $expiresAt = Carbon::parse($license->expires_at);

        // If license is already expired
        if ($now->greaterThan($expiresAt)) {
            return 'License expired';
        }

        // Get the diff
        $diff = $now->diff($expiresAt);

        return sprintf(
            '%d year(s), %d month(s), %d day(s), %d hour(s)',
            $diff->y,
            $diff->m,
            $diff->d,
            $diff->h
        );
    }
}

// Transaction helper function
function recordTransaction($date, $reason, $status, $amount, $accountId, $userId)
{
    // Find the account
    $account = Account::find($accountId);

    if (!$account) {
        throw new \Exception("Account ID {$accountId} not found.");
    }

    $sellingId = $account->selling_id;

    $beforeAmount = $account->balance;

    // Determine after balance
    $afterAmount = ($status === "Out")
        ? $beforeAmount - $amount
        : $beforeAmount + $amount;

    // Create transaction record
    $transaction = Transaction::create([
        'date'       => $date,
        'reason'     => $reason,
        'status'     => $status,
        'amount'     => $amount,
        'before'     => $beforeAmount,
        'after'      => $afterAmount,
        'account_id' => $accountId,
        'user_id'    => $userId,
        'selling_id' => $sellingId,
    ]);

    // Update account balance
    $account->update(['balance' => $afterAmount]);

    return $transaction;
}

if (!function_exists('unformatNumber')) {
    function unformatNumber($value)
    {
        return (int) str_replace([',', ' '], '', $value);
    }
}

if (!function_exists('getBatchNo')) {
    function getBatchNo($sellingId)
    {
        $profile = BusinessProfile::where('selling_id', $sellingId)->first();

        return $profile ? (int) $profile->batch_no : 0;
    }
}
